close all
clear
clc

colors = distinguishable_colors(30);
cyan = colors(11,:);
pink = colors(5,:);
green = colors(13,:);
purple = colors(16,:);
magenta = colors(15,:);
yellow = colors(6,:);

transparency = 0;

values = 0:1:3;
valueDiffs = -3:1:3;
positiveValueDiffs = 1:1:3;
nonNegativeValueDiffs = 0:1:3;
xLim = [-3.5 3.5];

sessions = 1:4;
parcodes = ['mar'; 'sol'; 'lua'; 'rio'; 'cor'; 'sal'; 'mel'; 'dia'; ...
    'ceu'; 'lei'; 'fio'; 'sul'; 'vai'; 'pai'; 'reu'; 'cai'; 'rei'; ...
    'dor'; 'giz'; 'mae'; 'tia'; 'pia'; 'fui'; 'gol'; 'gel'];

plotsFileBaseName = 'plots';
clocknum = clock;
dateString = [];
for i=1:5
    dateString = [dateString '_' num2str(clocknum(i))];
end
plotsFileName = [plotsFileBaseName dateString];

% Load all data.
[dataChoice, dataRt, dataValueLeft, dataValueRight, dataFixItem, ...
    dataFixTime, ~, ~] = GetValidDataTrials(parcodes, sessions, false, ...
    true);
dataValueLeft = (dataValueLeft ./ 5) + 3;
dataValueRight = (dataValueRight ./ 5) + 3;

% Load simulations from our model.
[simulChoice, simulRt, simulValueLeft, simulValueRight, simulFixItem, ...
    simulFixTime] = ConvertSimulationsCsvFiles('model1_expdata.csv', ...
    'model1_fixations.csv');

% Load simulations from the theta=1 model.
[ddmChoice, ddmRt, ddmValueLeft, ddmValueRight, ddmFixItem, ...
    ddmFixTime] = ConvertSimulationsCsvFiles('model3_expdata.csv', ...
    'model3_fixations.csv');


%% Choices.

% Plot actual choice curve.
[countDecisions, countLeftChosen] = Choices(valueDiffs, dataChoice, ...
    dataValueLeft, dataValueRight);
prob = countLeftChosen ./ countDecisions;
standardDev = sqrt(prob .* (1-prob));
confInt = Get95PercConfidenceInterval(standardDev, countDecisions);
figure; hold;
shadedErrorBar(valueDiffs, prob, confInt, {'-', 'Marker', '.', ...
    'MarkerSize', 20, 'Color', cyan, 'LineWidth', 3}, transparency);

% Plot simulated choice curve.
[countDecisions, countLeftChosen] = Choices(valueDiffs, simulChoice, ...
    simulValueLeft, simulValueRight);
prob = countLeftChosen ./ countDecisions;
standardDev = sqrt(prob .* (1-prob));
confInt = Get95PercConfidenceInterval(standardDev, countDecisions);
shadedErrorBar(valueDiffs, prob, confInt, {'-', 'Marker', '.', ...
    'MarkerSize', 20, 'Color', magenta, 'LineWidth', 3}, transparency);

% Plot DDM choice curve.
[countDecisions, countLeftChosen] = Choices(valueDiffs, ddmChoice, ...
    ddmValueLeft, ddmValueRight);
prob = countLeftChosen ./ countDecisions;
standardDev = sqrt(prob .* (1-prob));
confInt = Get95PercConfidenceInterval(standardDev, countDecisions);
errorbar(valueDiffs, prob, confInt, 'LineStyle', '--', 'Marker', '.', ...
    'MarkerSize', 20, 'Color', [0.5 0.5 0.5], 'LineWidth', 1);

line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5],'LineWidth', 2);
xlim(xLim);
ylim([0 1]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'fitted model', '\theta=1 model'}, 'FontSize', 12, ...
    'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Choices grouped by first fixation.

% Get data.
[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
    rightCountLeftChosen] = ChoicesGroupedByFirstFixation(...
    valueDiffs, dataChoice, dataFixItem, dataValueLeft, ...
	dataValueRight);

% Plot aggregated choice curve when left was 1st fixated.
figure; hold;
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* (1-leftProbLeftChosen));
confIntLeft = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
shadedErrorBar(valueDiffs, leftProbLeftChosen, confIntLeft, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

% Plot aggregated choice curve when right was 1st fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confIntRight = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
shadedErrorBar(valueDiffs, rightProbLeftChosen, confIntRight, ...
    {'--', 'Marker', 'o', 'MarkerSize', 6, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

% Get simulations.
[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
        rightCountLeftChosen] = ChoicesGroupedByFirstFixation(...
        valueDiffs, simulChoice, simulFixItem, simulValueLeft, ...
        simulValueRight);
    
% Plot simulated choice curves when left was 1st fixated.
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* (1-leftProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
shadedErrorBar(valueDiffs, leftProbLeftChosen, confInt, {'-', ...
    'Marker', '.', 'MarkerSize', 20, 'Color', magenta, ...
    'LineWidth', 3}, transparency);

% Plot simulated choice curves when right was 1st fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
shadedErrorBar(valueDiffs, rightProbLeftChosen, confInt, {'--', ...
    'Marker', 'o', 'MarkerSize', 6, 'Color', magenta, ...
    'LineWidth', 3}, transparency);

% Get DDM simulations.
[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
        rightCountLeftChosen] = ChoicesGroupedByFirstFixation(...
        valueDiffs, ddmChoice, ddmFixItem, ddmValueLeft, ddmValueRight);
    
% Plot DDM choice curves when left was 1st fixated.
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* (1-leftProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
errorbar(valueDiffs, leftProbLeftChosen, confInt, 'LineStyle', '-', ...
    'Marker', '.', 'MarkerSize', 20, 'Color', [0.5 0.5 0.5], ...
    'LineWidth', 1);

% Plot DDM choice curves when right was 1st fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
errorbar(valueDiffs, rightProbLeftChosen, confInt, 'LineStyle', '--', ...
    'Marker', 'o', 'MarkerSize', 6, 'Color', [0.5 0.5 0.5], ...
    'LineWidth', 1);

xlim(xLim);
ylim([0 1]);
line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5]);
xlabel('r_{left} - r_{right}', 'FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
set(gca,'FontSize',20);
legend({'data 1st left', 'data 1st right', 'fitted model 1st left', ...
    'fitted model 1st right', '\theta=1 model 1st left', ...
    '\theta=1 model 1st right'}, ...
    'FontSize', 12, 'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Choices grouped by last fixation.

% Get data.
[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
    rightCountLeftChosen] = ChoicesGroupedByLastFixation(valueDiffs, ...
    dataChoice, dataFixItem, dataValueLeft, dataValueRight);

% Plot aggregated choice curve when left was last fixated.
figure; hold;
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* (1-leftProbLeftChosen));
confIntLeft = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
shadedErrorBar(valueDiffs, leftProbLeftChosen, confIntLeft, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

% Plot aggregated choice curve when right was last fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confIntRight = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
shadedErrorBar(valueDiffs, rightProbLeftChosen, confIntRight, ...
    {'--', 'Marker', 'o', 'MarkerSize', 6, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

% Size of bias effect.
biasData = leftProbLeftChosen(4) - rightProbLeftChosen(4);

% Get simulations.
[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
    rightCountLeftChosen] = ChoicesGroupedByLastFixation(valueDiffs, ...
    simulChoice, simulFixItem, simulValueLeft, simulValueRight);

% Plot simulated choice curves when left was last fixated.
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* (1-leftProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
shadedErrorBar(valueDiffs, leftProbLeftChosen, confInt, {'-', 'Marker', ...
    '.', 'MarkerSize', 20, 'Color', magenta, 'LineWidth', 3}, ...
    transparency);

% Plot simulated choice curves when left was last fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
shadedErrorBar(valueDiffs, rightProbLeftChosen, confInt, {'--', ...
    'Marker', 'o', 'MarkerSize', 6, 'Color', magenta, 'LineWidth', 3}, ...
    transparency);

% Size of bias effect.
biasSimul = leftProbLeftChosen(4) - rightProbLeftChosen(4);

% Get DDM simulations.
[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
    rightCountLeftChosen] = ChoicesGroupedByLastFixation(valueDiffs, ...
    ddmChoice, ddmFixItem, ddmValueLeft, ddmValueRight);

% Plot simulated choice curves when left was last fixated.
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* (1-leftProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
errorbar(valueDiffs, leftProbLeftChosen, confInt, 'LineStyle', '-', ...
    'Marker', '.', 'MarkerSize', 20, 'Color', [0.5 0.5 0.5], ...
    'LineWidth', 1);

% Plot simulated choice curves when left was last fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
errorbar(valueDiffs, rightProbLeftChosen, confInt, 'LineStyle', '--', ...
    'Marker', 'o', 'MarkerSize', 6, 'Color', [0.5 0.5 0.5], ...
    'LineWidth', 1);

xlim(xLim);
ylim([0 1]);
line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5], ...
    'LineWidth', 2);
xlabel('r_{left} - r_{right}', 'FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
set(gca,'FontSize',20);
legend({'data last left', 'data last right', 'fitted model last left', ...
    'fitted model last right', '\theta=1 model last left', ...
    '\theta=1 model last right'}, 'FontSize', 12, 'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Choices grouped by longest fixation time.

% Get data.
[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
    rightCountLeftChosen] = ChoicesGroupedByTotalFixation( ...
    valueDiffs, dataChoice, dataFixItem, dataFixTime, dataValueLeft, ...
    dataValueRight);

% Plot aggregated choice curve when left most fixated.
figure; hold;
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* (1-leftProbLeftChosen));
confIntLeft = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
shadedErrorBar(valueDiffs, leftProbLeftChosen, confIntLeft, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

% Plot aggregated choice curve when right most fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confIntRight = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
shadedErrorBar(valueDiffs, rightProbLeftChosen, confIntRight, ...
    {'--', 'Marker', 'o', 'MarkerSize', 6, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

% Get simulations.
[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
    rightCountLeftChosen] = ChoicesGroupedByTotalFixation( ...
    valueDiffs, simulChoice, simulFixItem, simulFixTime, ...
    simulValueLeft, simulValueRight);

% Plot simulated choice curves when left was most fixated.
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* ...
    (1-leftProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
shadedErrorBar(valueDiffs, leftProbLeftChosen, confInt, {'-', ...
    'Marker', '.', 'MarkerSize', 20, 'Color', magenta, ...
    'LineWidth', 3}, transparency);

% Plot simulated choice curves when right was most fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
shadedErrorBar(valueDiffs, rightProbLeftChosen, confInt, {'--', ...
    'Marker', 'o', 'MarkerSize', 6, 'Color', magenta, ...
    'LineWidth', 3}, transparency);

% Get DDM simulations.
[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
    rightCountLeftChosen] = ChoicesGroupedByTotalFixation( ...
    valueDiffs, ddmChoice, ddmFixItem, ddmFixTime, ...
    ddmValueLeft, ddmValueRight);

% Plot simulated choice curves when left was most fixated.
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* ...
    (1-leftProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
errorbar(valueDiffs, leftProbLeftChosen, confInt, 'LineStyle', '-', ...
    'Marker', '.', 'MarkerSize', 20, 'Color', [0.5 0.5 0.5], ...
    'LineWidth', 1);

% Plot simulated choice curves when right was most fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
errorbar(valueDiffs, rightProbLeftChosen, confInt, 'LineStyle', '--', ...
    'Marker', 'o', 'MarkerSize', 6, 'Color', [0.5 0.5 0.5], ...
    'LineWidth', 1);

xlim(xLim);
ylim([0 1]);
line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5]);
xlabel('r_{left} - r_{right}', 'FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
set(gca,'FontSize',20);
legend({'data most left', 'data most right', 'fitted model most left', ...
    'fitted model most right', '\theta=1 model most left', ...
    '\theta=1 model most right'}, 'FontSize', 12, 'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Probability of choosing left item vs. time advantage of left item.

timeBins = -600:200:600;

% Get probabilities for data.
[countChoseLeft, countTotal] = ProbChooseLeftVsTimeAdvantageLeft( ...
    timeBins, dataChoice, dataFixItem, dataFixTime);

means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(countChoseLeft)
    probChoseLeft = countChoseLeft(j) / countTotal(j);
    means(j) = probChoseLeft;
    stdev = sqrt(probChoseLeft .* (1-probChoseLeft));
    confInts(j) = Get95PercConfidenceInterval(stdev, countTotal(j));
end

figure; hold;
h1 = bar(1:length(timeBins), means, 'FaceColor', 0.8 + 0.2 * cyan, ...
    'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, 'LineWidth', 3);
errorbar(1:length(timeBins), means, confInts, '.', 'Color', 0.7 * cyan, ...
    'LineWidth', 3);

% Get probabilities for simulations.
[countChoseLeft, countTotal] = ProbChooseLeftVsTimeAdvantageLeft( ...
    timeBins, simulChoice, simulFixItem, simulFixTime);

means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(countChoseLeft)
    probChoseLeft = countChoseLeft(j) / countTotal(j);
    means(j) = probChoseLeft;
    stdev = sqrt(probChoseLeft .* (1-probChoseLeft));
    confInts(j) = Get95PercConfidenceInterval(stdev, countTotal(j));
end
h2 = errorbar(1:length(timeBins), means, confInts, '--.', ...
    'MarkerSize', 25, 'Color', magenta, 'LineWidth', 3);

% Get probabilities for DDM simulations.
[countChoseLeft, countTotal] = ProbChooseLeftVsTimeAdvantageLeft( ...
    timeBins, ddmChoice, ddmFixItem, ddmFixTime);

means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(countChoseLeft)
    probChoseLeft = countChoseLeft(j) / countTotal(j);
    means(j) = probChoseLeft;
    stdev = sqrt(probChoseLeft .* (1-probChoseLeft));
    confInts(j) = Get95PercConfidenceInterval(stdev, countTotal(j));
end
h3 = errorbar(1:length(timeBins), means, confInts, '--.', ...
    'MarkerSize', 20, 'Color', [0.5 0.5 0.5], 'LineWidth', 1);

xlabel('Overall relative fixation left [ms]', 'FontSize', 24);
ylabel('P(choose left)','FontSize', 24);
xlim([0 length(timeBins)+1]);

xlabels = cell(length(timeBins), 1);
for i=1:length(timeBins)
    xlabels{i,1} = sprintf('%d', timeBins(i));
end
set(gca, 'XTick', 1:length(timeBins), 'XTickLabel', xlabels);
set(gca,'FontSize',20);
legend([h1 h2 h3], {'data', 'fitted model', '\theta=1 model'}, ...
    'FontSize', 12, 'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Corrected probability of choosing left item vs. time advantage of left
% item.

timeBins = -600:200:600;

% Get probabilities for data.
avgProbs = GetAvgProbChooseLeftByValueDiff(valueDiffs, dataValueLeft, ...
    dataValueRight, dataChoice);
corrProbs = CorrProbChooseLeftVsTimeAdvantageLeft(timeBins, ...
    valueDiffs, avgProbs, dataChoice, dataFixItem, dataFixTime, ...
    dataValueLeft, dataValueRight);
means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(corrProbs)
    means(j) = mean(corrProbs{j});
    stdev = std(corrProbs{j});
    confInts(j) = Get95PercConfidenceInterval(stdev, length(corrProbs{j}));
end

figure; hold;
h1 = bar(1:length(timeBins), means, 'FaceColor', 0.8 + 0.2 * cyan, ...
    'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, 'LineWidth', 3);
errorbar(1:length(timeBins), means, confInts, '.', 'Color', ...
    0.7 * cyan, 'LineWidth', 3);

% Get probabilities for simulations.
avgProbs = GetAvgProbChooseLeftByValueDiff(valueDiffs, simulValueLeft, ...
    simulValueRight, simulChoice);
corrProbs = CorrProbChooseLeftVsTimeAdvantageLeft(timeBins, ...
    valueDiffs, avgProbs, simulChoice, simulFixItem, simulFixTime, ...
    simulValueLeft, simulValueRight);
means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(corrProbs)
    means(j) = mean(corrProbs{j});
    stdev = std(corrProbs{j});
    confInts(j) = Get95PercConfidenceInterval(stdev, length(corrProbs{j}));
end

h2 = errorbar(1:length(timeBins), means, confInts, '--.', ...
    'MarkerSize', 25, 'Color', magenta, 'LineWidth', 3);

% Get probabilities for DDM simulations.
avgProbs = GetAvgProbChooseLeftByValueDiff(valueDiffs, ddmValueLeft, ...
    ddmValueRight, ddmChoice);
corrProbs = CorrProbChooseLeftVsTimeAdvantageLeft(timeBins, ...
    valueDiffs, avgProbs, ddmChoice, ddmFixItem, ddmFixTime, ...
    ddmValueLeft, ddmValueRight);
means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(corrProbs)
    means(j) = mean(corrProbs{j});
    stdev = std(corrProbs{j});
    confInts(j) = Get95PercConfidenceInterval(stdev, length(corrProbs{j}));
end

h3 = errorbar(1:length(timeBins), means, confInts, '--.', ...
    'MarkerSize', 20, 'Color', [0.5 0.5 0.5], 'LineWidth', 1);

xlabel('Overall relative fixation left [ms]', 'FontSize', 24);
ylabel('Corrected p(choose left)','FontSize', 24);
xlim([0 length(timeBins)+1]);

xlabels = cell(length(timeBins), 1);
for i=1:length(timeBins)
    xlabels{i,1} = sprintf('%d', timeBins(i));
end
set(gca, 'XTick', 1:length(timeBins), 'XTickLabel', xlabels);
set(gca,'FontSize',20);
legend([h1 h2 h3], {'data', 'fitted model', '\theta=1 model'}, ...
    'FontSize', 12, 'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Correlation value left minus value right vs. time advantage of left
% item.

% Data.
[valueDiff, timeAdvantageLeft] = ValueDifferenceVsTimeAdvantageLeft( ...
    dataChoice, dataFixItem, dataFixTime, dataValueLeft, dataValueRight);
figure; hold;
scatter(timeAdvantageLeft, valueDiff, 40, 'o', 'MarkerEdgeColor', cyan, ...
    'LineWidth', 1, 'MarkerEdgeAlpha', 0.5);

% Simulations.
[valueDiff, timeAdvantageLeft] = ValueDifferenceVsTimeAdvantageLeft( ...
    simulChoice, simulFixItem, simulFixTime, simulValueLeft, ...
    simulValueRight);
scatter(timeAdvantageLeft, valueDiff, 40, 'o', 'MarkerEdgeColor', ...
    magenta, 'LineWidth', 1, 'MarkerEdgeAlpha', 0.5);

% DDM simulations.
[valueDiff, timeAdvantageLeft] = ValueDifferenceVsTimeAdvantageLeft( ...
    ddmChoice, ddmFixItem, ddmFixTime, ddmValueLeft, ddmValueRight);
scatter(timeAdvantageLeft, valueDiff, 40, 'o', 'MarkerEdgeColor', ...
    [0.5 0.5 0.5], 'LineWidth', 1, 'MarkerEdgeAlpha', 0.5);

xlabel('Overall relative fixation left [ms]', 'FontSize', 24);
ylabel('r_{left} - r_{right}','FontSize', 24);
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Probability of choosing first seen item vs. first fixation duration.

timeBins = 0:200:600;

% Get probabilities for data.
[countChoseFirst, countTotal] = ProbChooseFirstVsTimeFirstFix( ...
    timeBins, dataChoice, dataFixItem, dataFixTime);
means = zeros(length(timeBins), 1);
confInts = zeros(length(timeBins), 1);
for j=1:length(countChoseFirst)
    probChoseFirst = countChoseFirst(j) / countTotal(j);
    means(j,1) = probChoseFirst;
    stdev = sqrt(probChoseFirst .* (1-probChoseFirst));
    confInts(j,1) = Get95PercConfidenceInterval(stdev, countTotal(j));
end

figure; hold;
h1 = bar(1:length(timeBins), means, 'FaceColor', 0.8 + 0.2 * cyan, ...
    'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, 'LineWidth', 3);
errorbar(1:length(timeBins), means, confInts, '.', 'Color', ...
    0.7 * cyan, 'LineWidth', 3);

% Get probabilities for simulations.
[countChoseFirst, countTotal] = ProbChooseFirstVsTimeFirstFix( ...
    timeBins, simulChoice, simulFixItem, simulFixTime);
means = zeros(length(timeBins), 1);
confInts = zeros(length(timeBins), 1);
for j=1:length(countChoseFirst)
    probChoseFirst = countChoseFirst(j) / countTotal(j);
    means(j,1) = probChoseFirst;
    stdev = sqrt(probChoseFirst .* (1-probChoseFirst));
    confInts(j,1) = Get95PercConfidenceInterval(stdev, countTotal(j));
end

h2 = errorbar(1:length(timeBins), means, confInts, '--.', ...
    'MarkerSize', 25, 'Color', magenta, 'LineWidth', 3);

% Get probabilities for DDM simulations.
[countChoseFirst, countTotal] = ProbChooseFirstVsTimeFirstFix( ...
    timeBins, ddmChoice, ddmFixItem, ddmFixTime);
means = zeros(length(timeBins), 1);
confInts = zeros(length(timeBins), 1);
for j=1:length(countChoseFirst)
    probChoseFirst = countChoseFirst(j) / countTotal(j);
    means(j,1) = probChoseFirst;
    stdev = sqrt(probChoseFirst .* (1-probChoseFirst));
    confInts(j,1) = Get95PercConfidenceInterval(stdev, countTotal(j));
end

h3 = errorbar(1:length(timeBins), means, confInts, '--.', ...
    'MarkerSize', 20, 'Color', [0.5 0.5 0.5], 'LineWidth', 1);

xlabel('First fixation duration [ms]', 'FontSize', 24);
ylabel('P(first seen chosen)','FontSize', 24);

xlabels = cell(length(timeBins), 1);
for i=1:length(timeBins)
    xlabels{i,1} = sprintf('%d', timeBins(i));
end
set(gca, 'XTick', 1:length(timeBins), 'XTickLabel', xlabels);
set(gca,'FontSize',20);
legend([h1 h2 h3], {'data', 'fitted model', '\theta=1 model'}, ...
    'FontSize', 12, 'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Corrected probability of choosing first seen item vs. first fixation
% duration.

timeBins = 0:200:600;

% Get probabilities for data.
avgProbs = GetAvgProbChooseFirstFixByValueDiff(valueDiffs, ...
    dataValueLeft, dataValueRight, dataChoice, dataFixItem, ...
    dataFixTime);
corrProbs = CorrProbChooseFirstFixVsTimeFirstFix(timeBins, ...
    valueDiffs, avgProbs, dataChoice, dataFixItem, dataFixTime, ...
    dataValueLeft, dataValueRight);
means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(corrProbs)
    means(j) = mean(corrProbs{j});
    stdev = std(corrProbs{j});
    confInts(j) = Get95PercConfidenceInterval(stdev, length(corrProbs{j}));
end

figure; hold;
h1 = bar(1:length(timeBins), means, 'FaceColor', 0.8 + 0.2 * cyan, ...
    'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, 'LineWidth', 3);
errorbar(1:length(timeBins), means, confInts, '.', 'Color', ...
    0.7 * cyan, 'LineWidth', 3);

% Get probabilities for simulations.
avgProbs = GetAvgProbChooseFirstFixByValueDiff(valueDiffs, ...
    simulValueLeft, simulValueRight, simulChoice, simulFixItem, ...
    simulFixTime);
corrProbs = CorrProbChooseFirstFixVsTimeFirstFix(timeBins, ...
    valueDiffs, avgProbs, simulChoice, simulFixItem, simulFixTime, ...
    simulValueLeft, simulValueRight);
means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(corrProbs)
    means(j) = mean(corrProbs{j});
    stdev = std(corrProbs{j});
    confInts(j) = Get95PercConfidenceInterval(stdev, length(corrProbs{j}));
end

h2 = errorbar(1:length(timeBins), means, confInts, '--.', ...
    'MarkerSize', 25, 'Color', magenta, 'LineWidth', 3);

% Get probabilities for DDM simulations.
avgProbs = GetAvgProbChooseFirstFixByValueDiff(valueDiffs, ...
    ddmValueLeft, ddmValueRight, ddmChoice, ddmFixItem, ...
    ddmFixTime);
corrProbs = CorrProbChooseFirstFixVsTimeFirstFix(timeBins, ...
    valueDiffs, avgProbs, ddmChoice, ddmFixItem, ddmFixTime, ...
    ddmValueLeft, ddmValueRight);
means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(corrProbs)
    means(j) = mean(corrProbs{j});
    stdev = std(corrProbs{j});
    confInts(j) = Get95PercConfidenceInterval(stdev, length(corrProbs{j}));
end

h3 = errorbar(1:length(timeBins), means, confInts, '--.', ...
    'MarkerSize', 20, 'Color', [0.5 0.5 0.5], 'LineWidth', 1);

xlabel('First fixation duration [ms]', 'FontSize', 24);
ylabel('Corrected p(first seen chosen)','FontSize', 24);

xlabels = cell(length(timeBins), 1);
for i=1:length(timeBins)
    xlabels{i,1} = sprintf('%d', timeBins(i));
end
set(gca, 'XTick', 1:length(timeBins), 'XTickLabel', xlabels);
set(gca,'FontSize',20);
legend([h1 h2 h3], {'data', 'fitted model', '\theta=1 model'}, ...
    'FontSize', 12, 'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Correlation value first fixated minus other value vs. first fixation
% duration.

% Data.
[valueDiff, timeAdvantageLeft] = ValueDifferenceVsFirstFixDuration( ...
    dataFixItem, dataFixTime, dataValueLeft, dataValueRight);
figure; hold;
scatter(timeAdvantageLeft, valueDiff, 40, 'o', 'MarkerEdgeColor', cyan, ...
    'LineWidth', 1, 'MarkerEdgeAlpha', 0.5);

% Simulations.
[valueDiff, timeAdvantageLeft] = ValueDifferenceVsFirstFixDuration( ...
    simulFixItem, simulFixTime, simulValueLeft, simulValueRight);
scatter(timeAdvantageLeft, valueDiff, 40, 'o', 'MarkerEdgeColor', ...
    magenta, 'LineWidth', 1, 'MarkerEdgeAlpha', 0.5);

% DDM simulations.
[valueDiff, timeAdvantageLeft] = ValueDifferenceVsFirstFixDuration( ...
    ddmFixItem, ddmFixTime, ddmValueLeft, ddmValueRight);
scatter(timeAdvantageLeft, valueDiff, 40, 'o', 'MarkerEdgeColor', ...
    [0.5 0.5 0.5], 'LineWidth', 1, 'MarkerEdgeAlpha', 0.5);

xlabel('First fixation duration [ms]', 'FontSize', 24);
set(gca,'FontSize',20);
ylabel('r_{first fix} - r_{other}','FontSize', 24);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% RTs vs. value difference.

% Plot actual reaction times.
reactionTimes = ReactionTimesByValueDiff(valueDiffs, dataRt, ...
    dataValueLeft, dataValueRight);
means = zeros(length(valueDiffs),1);
confInt = zeros(length(valueDiffs),1);
for j=1:length(valueDiffs)
    means(j) = mean(reactionTimes{j});
    confInt(j) = Get95PercConfidenceInterval(std(reactionTimes{j}), ...
        length(reactionTimes{j}));
end
figure; hold;
shadedErrorBar(valueDiffs, means, confInt, {'-', 'Marker', '.', ...
    'MarkerSize', 20, 'Color', cyan, 'LineWidth', 3}, transparency);

% Plot simulated reaction times.
reactionTimes = ReactionTimesByValueDiff(valueDiffs, simulRt, ...
    simulValueLeft, simulValueRight);
means = zeros(length(valueDiffs),1);
confInt = zeros(length(valueDiffs),1);
for j=1:length(valueDiffs)
    means(j) = mean(reactionTimes{j});
    confInt(j) = Get95PercConfidenceInterval(std(reactionTimes{j}), ...
        length(reactionTimes{j}));
end
shadedErrorBar(valueDiffs, means, confInt, {'-', 'Marker', '.', ...
    'MarkerSize', 20, 'Color', magenta, 'LineWidth', 3}, transparency);

% Plot DDM reaction times.
reactionTimes = ReactionTimesByValueDiff(valueDiffs, ddmRt, ...
    ddmValueLeft, ddmValueRight);
means = zeros(length(valueDiffs),1);
confInt = zeros(length(valueDiffs),1);
for j=1:length(valueDiffs)
    means(j) = mean(reactionTimes{j});
    confInt(j) = Get95PercConfidenceInterval(std(reactionTimes{j}), ...
        length(reactionTimes{j}));
end
errorbar(valueDiffs, means, confInt, 'LineStyle', '--', 'Marker', '.', ...
    'MarkerSize', 20, 'Color', [0.5 0.5 0.5], 'LineWidth', 1);

xlim(xLim);
ylim([0 3000]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('Reaction time [ms]','FontSize', 24);
set(gca, 'FontSize', 20);
legend({'data', 'fitted model', '\theta=1 model'}, ...
    'FontSize', 12, 'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% RTs vs. value best minus value worst.

% Plot actual reaction times.
reactionTimes = ReactionTimesVsValueBestMinusWorst( ...
    nonNegativeValueDiffs, dataRt, dataValueLeft, dataValueRight);
means = zeros(length(nonNegativeValueDiffs),1);
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    means(j) = mean(reactionTimes{j});
    confInt(j) = Get95PercConfidenceInterval(std(reactionTimes{j}), ...
        length(reactionTimes{j}));
end

figure; hold;
h1 = bar(nonNegativeValueDiffs, means, 'FaceColor', ...
    0.8 + 0.2 * cyan, 'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, ...
    'LineWidth', 3);
errorbar(nonNegativeValueDiffs, means, confInt, '.', ...
    'Color', 0.7 * cyan, 'LineWidth', 3);

% Plot simulated reaction times.
reactionTimes = ReactionTimesVsValueBestMinusWorst( ...
    nonNegativeValueDiffs, simulRt, simulValueLeft, simulValueRight);
means = zeros(length(nonNegativeValueDiffs),1);
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    means(j) = mean(reactionTimes{j});
    confInt(j) = Get95PercConfidenceInterval(std(reactionTimes{j}), ...
        length(reactionTimes{j}));
end
h2 = errorbar(nonNegativeValueDiffs, means, confInt, '--.', ...
    'MarkerSize', 25, 'Color', magenta, 'LineWidth', 3);

% Plot DDM reaction times.
reactionTimes = ReactionTimesVsValueBestMinusWorst( ...
    nonNegativeValueDiffs, ddmRt, ddmValueLeft, ddmValueRight);
means = zeros(length(nonNegativeValueDiffs),1);
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    means(j) = mean(reactionTimes{j});
    confInt(j) = Get95PercConfidenceInterval(std(reactionTimes{j}), ...
        length(reactionTimes{j}));
end
h3 = errorbar(nonNegativeValueDiffs, means, confInt, '--.', ...
    'MarkerSize', 20, 'Color', [0.5 0.5 0.5], 'LineWidth', 1);

xlabel('r_{closest} - r_{farthest}','FontSize', 24);
ylabel('Reaction time [ms]','FontSize', 24);

xlim([-0.5 3.5]);
xlabels = cell(length(nonNegativeValueDiffs), 1);
for i=1:length(nonNegativeValueDiffs)
    xlabels{i,1} = sprintf('%d', nonNegativeValueDiffs(i));
end
set(gca, 'XTick', nonNegativeValueDiffs, 'XTickLabel', xlabels);
set(gca, 'FontSize', 20);
legend([h1 h2 h3], {'data', 'fitted model', '\theta=1 model'}, ...
    'FontSize', 12, 'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Number of fixations vs. value best minus value worst.

% Plot actual number of fixations.
numFixations = NumFixationsVsValueBestMinusWorst(nonNegativeValueDiffs, ...
    dataFixItem, dataValueLeft, dataValueRight);
means = zeros(length(nonNegativeValueDiffs),1);
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    means(j) = mean(numFixations{j});
    confInt(j) = Get95PercConfidenceInterval(std(numFixations{j}), ...
        length(numFixations{j}));
end

figure; hold;
h1 = bar(nonNegativeValueDiffs, means, 'FaceColor', ...
    0.8 + 0.2 * cyan, 'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, ...
    'LineWidth', 3);
errorbar(nonNegativeValueDiffs, means, confInt, '.', ...
    'Color', 0.7 * cyan, 'LineWidth', 3);

% Plot simulated number of fixations.
numFixations = NumFixationsVsValueBestMinusWorst(nonNegativeValueDiffs, ...
    simulFixItem, simulValueLeft, simulValueRight);
means = zeros(length(nonNegativeValueDiffs),1);
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    means(j) = mean(numFixations{j});
    confInt(j) = Get95PercConfidenceInterval(std(numFixations{j}), ...
        length(numFixations{j}));
end
h2 = errorbar(nonNegativeValueDiffs, means, confInt, '--.', ...
    'MarkerSize', 25, 'Color', magenta, 'LineWidth', 3);

% Plot DDM number of fixations.
numFixations = NumFixationsVsValueBestMinusWorst(nonNegativeValueDiffs, ...
    ddmFixItem, ddmValueLeft, ddmValueRight);
means = zeros(length(nonNegativeValueDiffs),1);
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    means(j) = mean(numFixations{j});
    confInt(j) = Get95PercConfidenceInterval(std(numFixations{j}), ...
        length(numFixations{j}));
end
h3 = errorbar(nonNegativeValueDiffs, means, confInt, '--.', ...
    'MarkerSize', 20, 'Color', [0.5 0.5 0.5], 'LineWidth', 1);

xlim([-0.5 3.5]);
xlabels = cell(length(nonNegativeValueDiffs), 1);
for i=1:length(nonNegativeValueDiffs)
    xlabels{i,1} = sprintf('%d', nonNegativeValueDiffs(i));
end
set(gca, 'XTick', nonNegativeValueDiffs, 'XTickLabel', xlabels);

xlabel('r_{closest} - r_{farthest}','FontSize', 24);
ylabel('Number of fixations','FontSize', 24);
set(gca, 'FontSize', 20);
legend([h1 h2 h3], {'data', 'fitted model', '\theta=1 model'}, ...
    'FontSize', 12, 'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Probability of error vs. value best minus value worst.

% Plot actual probability of error.
[countError, countTotal] = ProbErrorGroupedVsValueBestMinusWorst( ...
    positiveValueDiffs, dataChoice, dataValueLeft, dataValueRight);
figure; hold;
probError = countError ./ countTotal;
stdProbError = sqrt(probError .* (1-probError));
confInt = Get95PercConfidenceInterval(stdProbError, countTotal);
shadedErrorBar(positiveValueDiffs, probError, confInt, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

% Plot simulated probability of error.
[countError, countTotal] = ProbErrorGroupedVsValueBestMinusWorst( ...
    positiveValueDiffs, simulChoice, simulValueLeft, simulValueRight);
probError = countError ./ countTotal;
stdProbError = sqrt(probError .* (1-probError));
confInt = Get95PercConfidenceInterval(stdProbError, countTotal);
shadedErrorBar(positiveValueDiffs, probError, confInt, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta, ...
    'LineWidth', 3}, transparency);

xlabel('r_{closest} - r_{farthest}','FontSize', 24);
ylabel('P(error)','FontSize', 24);

xlim([0.5 3.5]);
ylim([0 0.5]);
xlabels = cell(length(positiveValueDiffs), 1);
for i=1:length(positiveValueDiffs)
    xlabels{i,1} = sprintf('%d', positiveValueDiffs(i));
end
set(gca, 'XTick', positiveValueDiffs, 'XTickLabel', xlabels);
set(gca, 'FontSize', 20);
legend({'data', 'model'}, 'Location', 'NorthEast', 'FontSize', 12);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% RT histograms.

% RT histogram for real reaction times.
reactionTimes = ReactionTimesByValueDiff(valueDiffs, dataRt, ...
    dataValueLeft, dataValueRight);
rts = [];
for j=1:length(valueDiffs)
    rts = [rts; reactionTimes{j}];
end
figure; hold on;
[n,b] = hist(rts, 0:50:10000);
n = n / length(rts);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);

% RT histogram simulated reaction times.
reactionTimes = ReactionTimesByValueDiff(valueDiffs, simulRt, ...
    simulValueLeft, simulValueRight);
rts = [];
for j=1:length(valueDiffs)
    rts = [rts; reactionTimes{j}];
end
[n,b] = hist(rts, 0:50:10000);
n = n / length(rts);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);

xlabel('Reaction time [ms]','FontSize', 24);
ylabel('Relative frequency','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'FontSize', 12);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Fixation times vs. value best - value worst.

meanFirstFixTimes = zeros(length(nonNegativeValueDiffs),2);
confIntFirstFixTimes = zeros(length(nonNegativeValueDiffs),2);
meanSecondFixTimes = zeros(length(nonNegativeValueDiffs),2);
confIntSecondFixTimes = zeros(length(nonNegativeValueDiffs),2);
meanOtherFixTimes = zeros(length(nonNegativeValueDiffs),2);
confIntOtherFixTimes = zeros(length(nonNegativeValueDiffs),2);
meanLastFixTimes = zeros(length(nonNegativeValueDiffs),2);
confIntLastFixTimes = zeros(length(nonNegativeValueDiffs),2);

% Plot data fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueBestMinusWorst(nonNegativeValueDiffs, ...
    dataFixItem, dataFixTime, dataValueLeft, dataValueRight);
for i=1:length(nonNegativeValueDiffs)
    meanFirstFixTimes(i,1) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,1) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,1) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,1) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

% Plot simulated fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueBestMinusWorst(nonNegativeValueDiffs, ...
    simulFixItem, simulFixTime, simulValueLeft, simulValueRight);
for i=1:length(nonNegativeValueDiffs)
    meanFirstFixTimes(i,2) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,2) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,2) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,2) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

errorbar_groups(meanFirstFixTimes', confIntFirstFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    nonNegativeValueDiffs);
xlabel('r_{closest} - r_{farthest}', 'FontSize', 24);
ylabel('First fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanSecondFixTimes', confIntSecondFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    nonNegativeValueDiffs);
xlabel('r_{closest} - r_{farthest}', 'FontSize', 24);
ylabel('Second fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanOtherFixTimes', confIntOtherFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    nonNegativeValueDiffs);
xlabel('r_{closest} - r_{farthest}', 'FontSize', 24);
ylabel('Other middle fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanLastFixTimes', confIntLastFixTimes', 'bar_colors', ...
    [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    nonNegativeValueDiffs);
xlabel('r_{closest} - r_{farthest}', 'FontSize', 24);
ylabel('Last fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Fixation times vs. value fixated - value unfixated.

meanFirstFixTimes = zeros(length(valueDiffs),2);
confIntFirstFixTimes = zeros(length(valueDiffs),2);
meanSecondFixTimes = zeros(length(valueDiffs),2);
confIntSecondFixTimes = zeros(length(valueDiffs),2);
meanOtherFixTimes = zeros(length(valueDiffs),2);
confIntOtherFixTimes = zeros(length(valueDiffs),2);
meanLastFixTimes = zeros(length(valueDiffs),2);
confIntLastFixTimes = zeros(length(valueDiffs),2);

% Plot data fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, ...
    dataFixItem, dataFixTime, dataValueLeft, dataValueRight);
for i=1:length(valueDiffs)
    meanFirstFixTimes(i,1) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,1) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,1) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,1) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

% Plot simulated fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, ...
    simulFixItem, simulFixTime, simulValueLeft, simulValueRight);
for i=1:length(valueDiffs)
    meanFirstFixTimes(i,2) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,2) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,2) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,2) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

errorbar_groups(meanFirstFixTimes', confIntFirstFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    valueDiffs);
xlabel('r_{fixated} - r_{unfixated}', 'FontSize', 24);
ylabel('First fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanSecondFixTimes', confIntSecondFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    valueDiffs);
xlabel('r_{fixated} - r_{unfixated}', 'FontSize', 24);
ylabel('Second fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanOtherFixTimes', confIntOtherFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    valueDiffs);
xlabel('r_{fixated} - r_{unfixated}', 'FontSize', 24);
ylabel('Other middle fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanLastFixTimes', confIntLastFixTimes', 'bar_colors', ...
    [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    valueDiffs);
xlabel('r_{fixated} - r_{unfixated}', 'FontSize', 24);
ylabel('Last fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Fixation times vs. value fixated.

meanFirstFixTimes = zeros(length(values),2);
confIntFirstFixTimes = zeros(length(values),2);
meanSecondFixTimes = zeros(length(values),2);
confIntSecondFixTimes = zeros(length(values),2);
meanOtherFixTimes = zeros(length(values),2);
confIntOtherFixTimes = zeros(length(values),2);
meanLastFixTimes = zeros(length(values),2);
confIntLastFixTimes = zeros(length(values),2);

% Plot data fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFix(values, dataFixItem, dataFixTime, ...
    dataValueLeft, dataValueRight);
for i=1:length(values)
    meanFirstFixTimes(i,1) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,1) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,1) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,1) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

% Plot simulated fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFix(values, simulFixItem, simulFixTime, ...
    simulValueLeft, simulValueRight);
for i=1:length(values)
    meanFirstFixTimes(i,2) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,2) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,2) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,2) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

errorbar_groups(meanFirstFixTimes', confIntFirstFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    values);
xlabel('r_{fixated}', 'FontSize', 24);
ylabel('First fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanSecondFixTimes', confIntSecondFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    values);
xlabel('r_{fixated}', 'FontSize', 24);
ylabel('Second fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanOtherFixTimes', confIntOtherFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    values);
xlabel('r_{fixated}', 'FontSize', 24);
ylabel('Other middle fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanLastFixTimes', confIntLastFixTimes', 'bar_colors', ...
    [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    valueDiffs);
xlabel('r_{fixated}', 'FontSize', 24);
ylabel('Last fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Fixation times vs. value unfixated.

meanFirstFixTimes = zeros(length(values),2);
confIntFirstFixTimes = zeros(length(values),2);
meanSecondFixTimes = zeros(length(values),2);
confIntSecondFixTimes = zeros(length(values),2);
meanOtherFixTimes = zeros(length(values),2);
confIntOtherFixTimes = zeros(length(values),2);
meanLastFixTimes = zeros(length(values),2);
confIntLastFixTimes = zeros(length(values),2);

% Plot data fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueUnfix(values, dataFixItem, dataFixTime, ...
    dataValueLeft, dataValueRight);
for i=1:length(values)
    meanFirstFixTimes(i,1) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,1) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,1) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,1) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

% Plot simulated fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueUnfix(values, simulFixItem, simulFixTime, ...
    simulValueLeft, simulValueRight);
for i=1:length(values)
    meanFirstFixTimes(i,2) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,2) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,2) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,2) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

errorbar_groups(meanFirstFixTimes', confIntFirstFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    values);
xlabel('r_{unfixated}', 'FontSize', 24);
ylabel('First fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanSecondFixTimes', confIntSecondFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    values);
xlabel('r_{unfixated}', 'FontSize', 24);
ylabel('Second fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanOtherFixTimes', confIntOtherFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    values);
xlabel('r_{unfixated}', 'FontSize', 24);
ylabel('Other middle fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanLastFixTimes', confIntLastFixTimes', 'bar_colors', ...
    [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    valueDiffs);
xlabel('r_{unfixated}', 'FontSize', 24);
ylabel('Last fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'Location', 'SouthEast', 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Histogram of all types of fixations.

% Plot data fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, dataFixItem, ...
    dataFixTime, dataValueLeft, dataValueRight);

allDataFirstFixTime = [];
allDataSecondFixTime = [];
allDataOtherFixTime = [];
allDataLastFixTime = [];
for j=1:length(valueDiffs)
    allDataFirstFixTime = [allDataFirstFixTime; firstFixTimes{j}];
    allDataSecondFixTime = [allDataSecondFixTime; secondFixTimes{j}];
    allDataOtherFixTime = [allDataOtherFixTime; otherFixTimes{j}];
    allDataLastFixTime = [allDataLastFixTime; lastFixTimes{j}];
end

% Plot simulated fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, simulFixItem, ...
    simulFixTime, simulValueLeft, simulValueRight);

allSimulFirstFixTime = [];
allSimulSecondFixTime = [];
allSimulOtherFixTime = [];
allSimulLastFixTime = [];
for j=1:length(valueDiffs)
    allSimulFirstFixTime = [allSimulFirstFixTime; firstFixTimes{j}];
    allSimulSecondFixTime = [allSimulSecondFixTime; secondFixTimes{j}];
    allSimulOtherFixTime = [allSimulOtherFixTime; otherFixTimes{j}];
    allSimulLastFixTime = [allSimulLastFixTime; lastFixTimes{j}];
end

figure; hold on;
[n,b] = hist(allDataFirstFixTime, 0:10:5000);
n = n / length(allDataFirstFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);
[n,b] = hist(allSimulFirstFixTime, 0:10:5000);
n = n / length(allSimulFirstFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);
xlim([0 1500]);
xlabel('First fixation time [ms]','FontSize', 24);
ylabel('Relative frequency','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

figure; hold on;
[n,b] = hist(allDataSecondFixTime, 0:10:5000);
n = n / length(allDataSecondFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);
[n,b] = hist(allSimulSecondFixTime, 0:10:5000);
n = n / length(allSimulSecondFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);
xlim([0 1500]);
xlabel('Second fixation time [ms]','FontSize', 24);
ylabel('Relative frequency','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

figure; hold on;
[n,b] = hist(allDataOtherFixTime, 0:10:5000);
n = n / length(allDataOtherFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);
[n,b] = hist(allSimulOtherFixTime, 0:10:5000);
n = n / length(allSimulOtherFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);
xlim([0 1500]);
xlabel('Other middle fixation time [ms]','FontSize', 24);
ylabel('Relative frequency','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);

figure; hold on;
[n,b] = hist(allDataLastFixTime, 0:10:5000);
n = n / length(allDataLastFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);
[n,b] = hist(allSimulLastFixTime, 0:10:5000);
n = n / length(allSimulLastFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);
xlim([0 1500]);
xlabel('Last fixation time [ms]','FontSize', 24);
ylabel('Relative frequency','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'model'}, 'FontSize', 12);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Histogram of last fixations grouped by choice.

[dataLastFixTimesChoseFixed, dataLastFixTimesChoseUnfixed] = ...
    LastFixationDurationGroupedByChoice(dataChoice, dataFixItem, ...
    dataFixTime);

% Plot simulated fixation times.
[simulLastFixTimesChoseFixed, simulLastFixTimesChoseUnfixed] = ...
    LastFixationDurationGroupedByChoice(simulChoice, simulFixItem, ...
    simulFixTime);

figure; hold on;
[n,b] = hist(dataLastFixTimesChoseFixed, 0:10:1500);
n = n / sum(n);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);

[n,b] = hist(dataLastFixTimesChoseUnfixed, 0:10:1500);
n = n / sum(n);
plot(b, n, '--', 'Marker', 's', 'MarkerSize', 10, 'Color', cyan);

[n,b] = hist(simulLastFixTimesChoseFixed, 0:10:1500);
n = n / sum(n);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);

[n,b] = hist(simulLastFixTimesChoseUnfixed, 0:10:1500);
n = n / sum(n);
plot(b, n, '--', 'Marker', 's', 'MarkerSize', 10, 'Color', magenta);

xlim([0 1200]);
xlabel('Last fixation time [ms]','FontSize', 24);
ylabel('Relative frequency','FontSize', 24);
legend({'data chose fixed', 'data chose unfixed', 'model chose fixed', ...
    'model chose unfixed'}, 'FontSize', 12);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Probability of error vs value best minus value worst, grouped by choice.

% Data.
[countErrorChoseFixed, countTotalChoseFixed, countErrorChoseUnfixed, ...
    countTotalChoseUnfixed] = ProbErrorGroupedByChoice( ...
    nonNegativeValueDiffs, dataChoice, dataFixItem, dataValueLeft, ...
    dataValueRight);

figure; hold on;
probErrorChoseFixed = countErrorChoseFixed ./ countTotalChoseFixed;
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    standardDev = sqrt(probErrorChoseFixed(j) .* ...
        (1-probErrorChoseFixed(j)));
    confInt(j) = Get95PercConfidenceInterval(standardDev, ...
        countTotalChoseFixed(j));
end
errorbar(nonNegativeValueDiffs(2:end), probErrorChoseFixed(2:end), ...
    confInt(2:end), '-', 'Marker', '.', 'MarkerSize', 20, 'Color', ...
    cyan, 'LineWidth', 3);

probErrorChoseUnfixed = countErrorChoseUnfixed ./ countTotalChoseUnfixed;
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    standardDev = sqrt(probErrorChoseUnfixed(j) .* ...
        (1-probErrorChoseUnfixed(j)));
    confInt(j) = Get95PercConfidenceInterval(standardDev, ...
        countTotalChoseUnfixed(j));
end
errorbar(nonNegativeValueDiffs(2:end), probErrorChoseUnfixed(2:end), ...
    confInt(2:end), '--', 'Marker', 's', 'MarkerSize', 10, 'Color', ...
    cyan, 'LineWidth', 3);

% Simulations.
[countErrorChoseFixed, countTotalChoseFixed, countErrorChoseUnfixed, ...
    countTotalChoseUnfixed] = ProbErrorGroupedByChoice( ...
    nonNegativeValueDiffs, simulChoice, simulFixItem, simulValueLeft, ...
    simulValueRight);

probErrorChoseFixed = countErrorChoseFixed ./ countTotalChoseFixed;
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    standardDev = sqrt(probErrorChoseFixed(j) .* ...
        (1-probErrorChoseFixed(j)));
    confInt(j) = Get95PercConfidenceInterval(standardDev, ...
        countTotalChoseFixed(j));
end
errorbar(nonNegativeValueDiffs(2:end), probErrorChoseFixed(2:end), ...
    confInt(2:end), '-', 'Marker', '.', 'MarkerSize', 20, 'Color', ...
    magenta, 'LineWidth', 3);

probErrorChoseUnfixed = countErrorChoseUnfixed ./ countTotalChoseUnfixed;
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    standardDev = sqrt(probErrorChoseUnfixed(j) .* ...
        (1-probErrorChoseUnfixed(j)));
    confInt(j) = Get95PercConfidenceInterval(standardDev, ...
        countTotalChoseUnfixed(j));
end
errorbar(nonNegativeValueDiffs(2:end), probErrorChoseUnfixed(2:end), ...
    confInt(2:end), '--', 'Marker', 's', 'MarkerSize', 10, 'Color', ...
    magenta, 'LineWidth', 3);

xlabel('r_{closest} - r_{farthest}','FontSize', 24);
ylabel('P(error)','FontSize', 24);
set(gca,'FontSize',20);
legend({'data chose fixed', 'data chose unfixed', 'model chose fixed', ...
    'model chose unfixed'}, 'FontSize', 12);
xlim([0.5 3.5]);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Number of fixations by value difference, grouped by whether last fixated
% item was chosen.

% Data.
[dataNumFixationsChoseFixed, dataNumFixationsChoseUnfixed] = ...
    NumFixationsByChoiceAndValueDiff(nonNegativeValueDiffs, ...
    dataChoice, dataFixItem, dataFixTime, dataValueLeft, dataValueRight);

% Simulations.
[simulNumFixationsChoseFixed, simulNumFixationsChoseUnfixed] = ...
    NumFixationsByChoiceAndValueDiff(nonNegativeValueDiffs, ...
    simulChoice, simulFixItem, simulFixTime, simulValueLeft, ...
    simulValueRight);

means = zeros(length(nonNegativeValueDiffs),4);
for j=1:length(nonNegativeValueDiffs)
    means(j,1) = mean(dataNumFixationsChoseFixed{j});
    means(j,2) = mean(dataNumFixationsChoseUnfixed{j});
    means(j,3) = mean(simulNumFixationsChoseFixed{j});
    means(j,4) = mean(simulNumFixationsChoseUnfixed{j});
end

confInts = zeros(length(nonNegativeValueDiffs),4);
for j=1:length(nonNegativeValueDiffs)
    confInts(j,1) = Get95PercConfidenceInterval( ...
        std(dataNumFixationsChoseFixed{j}), ...
        length(dataNumFixationsChoseFixed{j}));
    confInts(j,2) = Get95PercConfidenceInterval( ...
        std(dataNumFixationsChoseUnfixed{j}), ...
        length(dataNumFixationsChoseUnfixed{j}));
    confInts(j,3) = Get95PercConfidenceInterval( ...
        std(simulNumFixationsChoseFixed{j}), ...
        length(simulNumFixationsChoseFixed{j}));
    confInts(j,4) = Get95PercConfidenceInterval( ...
        std(simulNumFixationsChoseUnfixed{j}), ...
        length(simulNumFixationsChoseUnfixed{j}));
end

errorbar_groups(means', confInts', 'bar_colors', ...
    [0.2 + 0.8 * cyan; 0.8 + 0.2 * cyan; 0.2 + 0.8 * magenta; ...
    0.8 + 0.2 * magenta;], 'bar_names', nonNegativeValueDiffs);
xlabel('r_{closest} - r_{farthest}', 'FontSize', 24);
ylabel('Number of fixations','FontSize', 24);
set(gca,'FontSize',20);
legend({'data chose last fix', 'data chose other', ...
    'model chose last fix', 'model chose other'}, ...
    'FontSize', 12, 'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Number of trials by value difference, grouped by whether last fixated
% item was chosen.

% Data.
[dataNumTrialsChoseFixed, dataNumTrialsChoseUnfixed] = ...
    TrialCountByChoiceAndValueDiff(nonNegativeValueDiffs, ...
    dataChoice, dataFixItem, dataValueLeft, dataValueRight);

% Simulations.
[simulNumTrialsChoseFixed, simulNumTrialsChoseUnfixed] = ...
    TrialCountByChoiceAndValueDiff(nonNegativeValueDiffs, ...
    simulChoice, simulFixItem, simulValueLeft, simulValueRight);

values = [dataNumTrialsChoseFixed, dataNumTrialsChoseUnfixed, ...
    simulNumTrialsChoseFixed, simulNumTrialsChoseUnfixed];

errorbar_groups(values', zeros(4,4), 'bar_colors', ...
    [0.2 + 0.8 * cyan; 0.8 + 0.2 * cyan; 0.2 + 0.8 * magenta; ...
    0.8 + 0.2 * magenta;], 'bar_names', nonNegativeValueDiffs);
xlabel('r_{closest} - r_{farthest}', 'FontSize', 24);
ylabel('Number of trials','FontSize', 24);
set(gca,'FontSize',20);
legend({'data chose last fix', 'data chose other', ...
    'model chose last fix', 'model chose other'}, ...
    'FontSize', 12, 'Location', 'NorthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Plot bar graph of fixation times for each type of fixation.

means = zeros(4,3);
confInts = zeros(4,3);

% Plot data fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, dataFixItem, ...
    dataFixTime, dataValueLeft, dataValueRight);
dataFirstFixTimes = [];
dataSecondFixTimes = [];
dataOtherFixTimes = [];
dataLastFixTimes = [];
for j=1:length(valueDiffs)
    dataFirstFixTimes = [dataFirstFixTimes; firstFixTimes{j,1}];
    dataSecondFixTimes = [dataSecondFixTimes; secondFixTimes{j,1}];
    dataOtherFixTimes = [dataOtherFixTimes; otherFixTimes{j,1}];
    dataLastFixTimes = [dataLastFixTimes; lastFixTimes{j,1}];
end
means(1,1) = mean(dataFirstFixTimes);
means(2,1) = mean(dataSecondFixTimes);
means(3,1) = mean(dataOtherFixTimes);
means(4,1) = mean(dataLastFixTimes);
confInts(1,1) = Get95PercConfidenceInterval(std(dataFirstFixTimes), ...
    length(dataFirstFixTimes));
confInts(2,1) = Get95PercConfidenceInterval(std(dataSecondFixTimes), ...
    length(dataSecondFixTimes));
confInts(3,1) = Get95PercConfidenceInterval(std(dataOtherFixTimes), ...
    length(dataOtherFixTimes));
confInts(4,1) = Get95PercConfidenceInterval(std(dataLastFixTimes), ...
    length(dataLastFixTimes));

% Plot simulated fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, simulFixItem, ...
    simulFixTime, simulValueLeft, simulValueRight);
simulFirstFixTimes = [];
simulSecondFixTimes = [];
simulOtherFixTimes = [];
simulLastFixTimes = [];
for j=1:length(valueDiffs)
    simulFirstFixTimes = [simulFirstFixTimes; firstFixTimes{j,1}];
    simulSecondFixTimes = [simulSecondFixTimes; secondFixTimes{j,1}];
    simulOtherFixTimes = [simulOtherFixTimes; otherFixTimes{j,1}];
    simulLastFixTimes = [simulLastFixTimes; lastFixTimes{j,1}];
end
means(1,2) = mean(simulFirstFixTimes);
means(2,2) = mean(simulSecondFixTimes);
means(3,2) = mean(simulOtherFixTimes);
means(4,2) = mean(simulLastFixTimes);
confInts(1,2) = Get95PercConfidenceInterval(std(simulFirstFixTimes), ...
    length(simulFirstFixTimes));
confInts(2,2) = Get95PercConfidenceInterval( ...
    std(simulSecondFixTimes), length(simulSecondFixTimes));
confInts(3,2) = Get95PercConfidenceInterval( ...
    std(simulOtherFixTimes), length(simulOtherFixTimes));
confInts(4,2) = Get95PercConfidenceInterval(std(simulLastFixTimes), ...
    length(simulLastFixTimes));

% Plot DDM fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, ddmFixItem, ...
    ddmFixTime, ddmValueLeft, ddmValueRight);
ddmFirstFixTimes = [];
ddmSecondFixTimes = [];
ddmOtherFixTimes = [];
ddmLastFixTimes = [];
for j=1:length(valueDiffs)
    ddmFirstFixTimes = [ddmFirstFixTimes; firstFixTimes{j,1}];
    ddmSecondFixTimes = [ddmSecondFixTimes; secondFixTimes{j,1}];
    ddmOtherFixTimes = [ddmOtherFixTimes; otherFixTimes{j,1}];
    ddmLastFixTimes = [ddmLastFixTimes; lastFixTimes{j,1}];
end
means(1,3) = mean(ddmFirstFixTimes);
means(2,3) = mean(ddmSecondFixTimes);
means(3,3) = mean(ddmOtherFixTimes);
means(4,3) = mean(ddmLastFixTimes);
confInts(1,3) = Get95PercConfidenceInterval(std(ddmFirstFixTimes), ...
    length(ddmFirstFixTimes));
confInts(2,3) = Get95PercConfidenceInterval( ...
    std(ddmSecondFixTimes), length(ddmSecondFixTimes));
confInts(3,3) = Get95PercConfidenceInterval( ...
    std(ddmOtherFixTimes), length(ddmOtherFixTimes));
confInts(4,3) = Get95PercConfidenceInterval(std(ddmLastFixTimes), ...
    length(ddmLastFixTimes));

errorbar_groups(means', confInts', 'bar_colors', ...
    [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta; 0.8 + 0.2 * [0.5 0.5 0.5]], ...
    'bar_names', 1:3);
set(gca, 'XTick', 1:12, 'XTickLabel', {'', '1st', '', '', '2nd', '', ...
    '', 'Other', '', '', 'Last', ''});
ylabel('Fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend({'data', 'fitted model', '\theta=1 model'}, 'FontSize', 12);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Probability that last fixation is to chosen item vs. value of last
% fixated item minus other value.

% Plot data.
[countLastFixToChosen, countTotal] = ...
    ProbLastFixIsToChosenItemVsValueLastSeenMinusOther(valueDiffs, ...
    dataFixItem, dataChoice, dataValueLeft, dataValueRight);
probLastFixToChosen = countLastFixToChosen ./ countTotal;
stdProbLastFixToChosen = sqrt(probLastFixToChosen .* ...
    (1-probLastFixToChosen));
confInt = Get95PercConfidenceInterval(stdProbLastFixToChosen, ...
    countTotal);
figure; hold;
shadedErrorBar(valueDiffs, probLastFixToChosen, confInt, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

% Size of bias effect.
bias = probLastFixToChosen(4) - 0.5;

% Plot simulations.
[countLastFixToChosen, countTotal] = ...
    ProbLastFixIsToChosenItemVsValueLastSeenMinusOther(valueDiffs, ...
    simulFixItem, simulChoice, simulValueLeft, simulValueRight);
probLastFixToChosen = countLastFixToChosen ./ countTotal;
stdProbLastFixToChosen = sqrt(probLastFixToChosen .* ...
    (1-probLastFixToChosen));
confInt = Get95PercConfidenceInterval(stdProbLastFixToChosen, countTotal);
shadedErrorBar(valueDiffs, probLastFixToChosen, confInt, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta, ...
    'LineWidth', 3}, transparency);

% Plot DDM simulations.
[countLastFixToChosen, countTotal] = ...
    ProbLastFixIsToChosenItemVsValueLastSeenMinusOther(valueDiffs, ...
    ddmFixItem, ddmChoice, ddmValueLeft, ddmValueRight);
probLastFixToChosen = countLastFixToChosen ./ countTotal;
stdProbLastFixToChosen = sqrt(probLastFixToChosen .* ...
    (1-probLastFixToChosen));
confInt = Get95PercConfidenceInterval(stdProbLastFixToChosen, countTotal);
errorbar(valueDiffs, probLastFixToChosen, confInt, ...
    '--', 'Marker', '.', 'MarkerSize', 20, 'Color', [0.5 0.5 0.5], ...
    'LineWidth', 1);

line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5], ...
    'LineWidth', 2);
line([0 0],[0 1],'LineStyle','--','Color',[0.5 0.5 0.5], 'LineWidth', 2);
xlim(xLim);
ylim([0 1]);
xlabel('r_{last seen item} - r_{other item}','FontSize', 24);
ylabel('P(choose last fixated item)','FontSize', 24);
% legend('Data', 'Model', 'Location', 'SouthEast');
set(gca,'FontSize',20);
legend({'data', 'fitted model', '\theta=1 model'}, ...
    'FontSize', 12, 'Location', 'SouthEast');

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Time advantage of item B over item A (last fixation was to A) before
% last fixation vs last fixation duration.

timeBinLimits = 0:300:900;

% Plot data.
timeAdvantage = TimeAdvantageUnfixItemVsLastFixTime(timeBinLimits, ...
    dataFixItem, dataFixTime, dataChoice);
means = zeros(length(timeBinLimits), 1);
confInts = zeros(length(timeBinLimits), 1);
for i=1:length(timeBinLimits)
   means(i,1) = mean(timeAdvantage{i});
   confInts(i,1) = Get95PercConfidenceInterval(std(timeAdvantage{i}), ...
       length(timeAdvantage{i}));
end
xlabels = cell(length(timeBinLimits),1);
for i=1:length(timeBinLimits)
    xlabels{i,1} = sprintf('%d', timeBinLimits(i));
end

figure; hold on;
h1 = bar(means, 'FaceColor', 0.8 + 0.2 * cyan, 'EdgeColor', ...
    0.7 * cyan, 'BarWidth', 0.4, 'LineWidth', 3);
errorbar(means, confInts, '.', 'Color', 0.7 * cyan, 'LineWidth', 3);
set(gca, 'XTick', 1:length(timeBinLimits), 'XTickLabel', xlabels);

% Plot simulations.
timeAdvantage = TimeAdvantageUnfixItemVsLastFixTime(timeBinLimits, ...
    simulFixItem, simulFixTime, simulChoice);
means = zeros(length(timeBinLimits), 1);
confInts = zeros(length(timeBinLimits), 1);
for i=1:length(timeBinLimits)
   means(i,1) = mean(timeAdvantage{i});
   confInts(i,1) = Get95PercConfidenceInterval(std(timeAdvantage{i}), ...
       length(timeAdvantage{i}));
end

h2 = errorbar(means, confInts, '--.', 'MarkerSize', 25, ...
    'Color', magenta, 'LineWidth', 3);

% Plot DDM simulations.
timeAdvantage = TimeAdvantageUnfixItemVsLastFixTime(timeBinLimits, ...
    ddmFixItem, ddmFixTime, ddmChoice);
means = zeros(length(timeBinLimits), 1);
confInts = zeros(length(timeBinLimits), 1);
for i=1:length(timeBinLimits)
   means(i,1) = mean(timeAdvantage{i});
   confInts(i,1) = Get95PercConfidenceInterval(std(timeAdvantage{i}), ...
       length(timeAdvantage{i}));
end

h3 = errorbar(means, confInts, '--.', 'MarkerSize', 20, ...
    'Color', [0.5 0.5 0.5], 'LineWidth', 1);

xlabel('Last fixation duration [ms]','FontSize', 24);
ylabel('Relative fixation non-chosen item [ms]','FontSize', 24);
set(gca,'FontSize',20);
legend([h1 h2 h3], {'data', 'fitted model', '\theta=1 model'}, ...
    'FontSize', 12, 'Location', 'SouthWest');

orient landscape
print('-dpsc2', '-append', plotsFileName);


close all
