close all
clear
clc

colors = distinguishable_colors(30);
cyan = colors(11,:);
pink = colors(5,:);
green = colors(13,:);
purple = colors(16,:);
magenta = colors(15,:);
yellow = colors(6,:);

transparency = 0;

values = 0:1:3;
valueDiffs = -3:1:3;
positiveValueDiffs = 1:1:3;
nonNegativeValueDiffs = 0:1:3;
xLim = [-3.5 3.5];

sessions = 1:4;
parcodes = ['mar'; 'sol'; 'lua'; 'rio'; 'cor'; 'sal'; 'mel'; 'dia'; ...
    'ceu'; 'lei'; 'fio'; 'sul'; 'vai'; 'pai'; 'reu'; 'cai'; 'rei'; ...
    'dor'; 'giz'; 'mae'; 'tia'; 'pia'; 'fui'; 'gol'; 'gel'];

plotsFileBaseName = 'plots';
clocknum = clock;
dateString = [];
for i=1:5
    dateString = [dateString '_' num2str(clocknum(i))];
end
plotsFileName = [plotsFileBaseName dateString];

% Load early memory trials.
[earlyChoice, earlyRt, earlyValueLeft, earlyValueRight, earlyFixItem, ...
    earlyFixTime, ~, ~] = GetValidDataTrials(parcodes, sessions, false, ...
    false, true, false);
earlyValueLeft = (earlyValueLeft ./ 5) + 3;
earlyValueRight = (earlyValueRight ./ 5) + 3;

% Load late memory trials.
[lateChoice, lateRt, lateValueLeft, lateValueRight, lateFixItem, ...
    lateFixTime, ~, ~] = GetValidDataTrials(parcodes, sessions, false, ...
    false, false, true);
lateValueLeft = (lateValueLeft ./ 5) + 3;
lateValueRight = (lateValueRight ./ 5) + 3;



%% Choices.

% Plot early trials choice curve.
[countDecisions, countLeftChosen] = Choices(valueDiffs, earlyChoice, ...
    earlyValueLeft, earlyValueRight);
prob = countLeftChosen ./ countDecisions;
standardDev = sqrt(prob .* (1-prob));
confInt = Get95PercConfidenceInterval(standardDev, countDecisions);
figure; hold;
shadedErrorBar(valueDiffs, prob, confInt, {'-', 'Marker', '.', ...
    'MarkerSize', 20, 'Color', cyan, 'LineWidth', 3}, transparency);

% Plot late trials choice curve.
[countDecisions, countLeftChosen] = Choices(valueDiffs, lateChoice, ...
    lateValueLeft, lateValueRight);
prob = countLeftChosen ./ countDecisions;
standardDev = sqrt(prob .* (1-prob));
confInt = Get95PercConfidenceInterval(standardDev, countDecisions);
shadedErrorBar(valueDiffs, prob, confInt, {'-', 'Marker', '.', ...
    'MarkerSize', 20, 'Color', magenta, 'LineWidth', 3}, transparency);
line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5],'LineWidth', 2);
xlim(xLim);
ylim([0 1]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Choices grouped by first fixation.

[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
    rightCountLeftChosen] = ChoicesGroupedByFirstFixation(...
    valueDiffs, earlyChoice, earlyFixItem, earlyValueLeft, ...
	earlyValueRight);

% Plot early trials choice curve when left was 1st fixated.
figure; hold;
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* (1-leftProbLeftChosen));
confIntLeft = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
shadedErrorBar(valueDiffs, leftProbLeftChosen, confIntLeft, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan}, transparency);

% Plot early trials choice curve when right was 1st fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confIntRight = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
shadedErrorBar(valueDiffs, rightProbLeftChosen, confIntRight, ...
    {'--', 'Marker', 'o', 'MarkerSize', 6, 'Color', cyan}, transparency);

[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
        rightCountLeftChosen] = ChoicesGroupedByFirstFixation(...
        valueDiffs, lateChoice, lateFixItem, lateValueLeft, ...
        lateValueRight);
    
% Plot late trials choice curves when left was 1st fixated.
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* (1-leftProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
shadedErrorBar(valueDiffs, leftProbLeftChosen, confInt, {'-', ...
    'Marker', '.', 'MarkerSize', 20, 'Color', magenta}, transparency);

% Plot late trials choice curves when right was 1st fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
shadedErrorBar(valueDiffs, rightProbLeftChosen, confInt, {'--', ...
    'Marker', 'o', 'MarkerSize', 6, 'Color', magenta}, transparency);

xlim(xLim);
ylim([0 1]);
line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5]);
xlabel('r_{left} - r_{right}', 'FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
legend('Early first fix left', 'Early first fix right', ...
    'Late first fix left', 'Late first fix right', 'Location', ...
    'SouthEast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Choices grouped by last fixation.

[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
    rightCountLeftChosen] = ChoicesGroupedByLastFixation(valueDiffs, ...
    earlyChoice, earlyFixItem, earlyValueLeft, earlyValueRight);

% Plot early trials choice curve when left was last fixated.
figure; hold;
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* (1-leftProbLeftChosen));
confIntLeft = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
shadedErrorBar(valueDiffs, leftProbLeftChosen, confIntLeft, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

% Plot aggregated choice curve when right was last fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confIntRight = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
shadedErrorBar(valueDiffs, rightProbLeftChosen, confIntRight, ...
    {'--', 'Marker', 'o', 'MarkerSize', 6, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

% Size of bias effect.
biasData = leftProbLeftChosen(4) - rightProbLeftChosen(4);

[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
    rightCountLeftChosen] = ChoicesGroupedByLastFixation(valueDiffs, ...
    lateChoice, lateFixItem, lateValueLeft, lateValueRight);

% Plot late trials choice curves when left was last fixated.
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* (1-leftProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
shadedErrorBar(valueDiffs, leftProbLeftChosen, confInt, {'-', 'Marker', ...
    '.', 'MarkerSize', 20, 'Color', magenta, 'LineWidth', 3}, ...
    transparency);

% Plot simulated choice curves when left was last fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
shadedErrorBar(valueDiffs, rightProbLeftChosen, confInt, {'--', ...
    'Marker', 'o', 'MarkerSize', 6, 'Color', magenta, 'LineWidth', 3}, ...
    transparency);

% Size of bias effect.
biasSimul = leftProbLeftChosen(4) - rightProbLeftChosen(4);

xlim(xLim);
ylim([0 1]);
line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5], ...
    'LineWidth', 2);
xlabel('r_{left} - r_{right}', 'FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
legend('Early last fix left', 'Early last fix right', ...
    'Late last fix left', 'Late last fix right', 'Location', ...
    'SouthEast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Choices grouped by longest fixation time.

[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
    rightCountLeftChosen] = ChoicesGroupedByTotalFixation( ...
    valueDiffs, earlyChoice, earlyFixItem, earlyFixTime, earlyValueLeft, ...
    earlyValueRight);

% Plot early trials choice curve when left most fixated.
figure; hold;
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* (1-leftProbLeftChosen));
confIntLeft = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
shadedErrorBar(valueDiffs, leftProbLeftChosen, confIntLeft, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan}, transparency);

% Plot early trials choice curve when right most fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confIntRight = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
shadedErrorBar(valueDiffs, rightProbLeftChosen, confIntRight, ...
    {'--', 'Marker', 'o', 'MarkerSize', 6, 'Color', cyan}, transparency);

[leftCountDecisions, leftCountLeftChosen, rightCountDecisions, ...
    rightCountLeftChosen] = ChoicesGroupedByTotalFixation( ...
    valueDiffs, lateChoice, lateFixItem, lateFixTime, ...
    lateValueLeft, lateValueRight);

% Plot late trials choice curves when left was most fixated.
leftProbLeftChosen = leftCountLeftChosen ./ leftCountDecisions;
stdLeftProbLeftChosen = sqrt(leftProbLeftChosen .* ...
    (1-leftProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdLeftProbLeftChosen, ...
    leftCountDecisions);
shadedErrorBar(valueDiffs, leftProbLeftChosen, confInt, {'-', ...
    'Marker', '.', 'MarkerSize', 20, 'Color', magenta}, transparency);

% Plot late trials choice curves when right was most fixated.
rightProbLeftChosen = rightCountLeftChosen ./ rightCountDecisions;
stdRightProbLeftChosen = sqrt(rightProbLeftChosen .* ...
    (1-rightProbLeftChosen));
confInt = Get95PercConfidenceInterval(stdRightProbLeftChosen, ...
    rightCountDecisions);
shadedErrorBar(valueDiffs, rightProbLeftChosen, confInt, {'--', ...
    'Marker', 'o', 'MarkerSize', 6, 'Color', magenta}, transparency);

xlim(xLim);
ylim([0 1]);
line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5]);
xlabel('r_{left} - r_{right}', 'FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
legend('Early most fix left', 'Early most fix right', ...
    'Late most fix left', 'Late most fix right', 'Location', ...
    'SouthEast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Probability of choosing left item vs. time advantage of left item.

timeBins = -600:200:600;

[countChoseLeft, countTotal] = ProbChooseLeftVsTimeAdvantageLeft( ...
    timeBins, earlyChoice, earlyFixItem, earlyFixTime);

means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(countChoseLeft)
    probChoseLeft = countChoseLeft(j) / countTotal(j);
    means(j) = probChoseLeft;
    stdev = sqrt(probChoseLeft .* (1-probChoseLeft));
    confInts(j) = Get95PercConfidenceInterval(stdev, countTotal(j));
end

figure; hold;
h1 = bar(1:length(timeBins), means, 'FaceColor', 0.8 + 0.2 * cyan, ...
    'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, 'LineWidth', 3);
errorbar(1:length(timeBins), means, confInts, '.', 'Color', 0.7 * cyan, ...
    'LineWidth', 3);

[countChoseLeft, countTotal] = ProbChooseLeftVsTimeAdvantageLeft( ...
    timeBins, lateChoice, lateFixItem, lateFixTime);

means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(countChoseLeft)
    probChoseLeft = countChoseLeft(j) / countTotal(j);
    means(j) = probChoseLeft;
    stdev = sqrt(probChoseLeft .* (1-probChoseLeft));
    confInts(j) = Get95PercConfidenceInterval(stdev, countTotal(j));
end
h2 = errorbar(1:length(timeBins), means, confInts, '--.', ...
    'MarkerSize', 25, 'Color', magenta, 'LineWidth', 3);

xlabel('Overall relative fixation left [ms]', 'FontSize', 24);
ylabel('P(choose left)','FontSize', 24);
legend([h1 h2], {'Early trials', 'Late trials'}, 'Location', 'SouthEast');
xlim([0 length(timeBins)+1]);

xlabels = cell(length(timeBins), 1);
for i=1:length(timeBins)
    xlabels{i,1} = sprintf('%d', timeBins(i));
end
set(gca, 'XTick', 1:length(timeBins), 'XTickLabel', xlabels);
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Corrected probability of choosing left item vs. time advantage of left
% item.

timeBins = -600:200:600;

avgProbs = GetAvgProbChooseLeftByValueDiff(valueDiffs, earlyValueLeft, ...
    earlyValueRight, earlyChoice);
corrProbs = CorrProbChooseLeftVsTimeAdvantageLeft(timeBins, ...
    valueDiffs, avgProbs, earlyChoice, earlyFixItem, earlyFixTime, ...
    earlyValueLeft, earlyValueRight);
means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(corrProbs)
    means(j) = mean(corrProbs{j});
    stdev = std(corrProbs{j});
    confInts(j) = Get95PercConfidenceInterval(stdev, length(corrProbs{j}));
end

figure; hold;
h1 = bar(1:length(timeBins), means, 'FaceColor', 0.8 + 0.2 * cyan, ...
    'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, 'LineWidth', 3);
errorbar(1:length(timeBins), means, confInts, '.', 'Color', ...
    0.7 * cyan, 'LineWidth', 3);

avgProbs = GetAvgProbChooseLeftByValueDiff(valueDiffs, lateValueLeft, ...
    lateValueRight, lateChoice);
corrProbs = CorrProbChooseLeftVsTimeAdvantageLeft(timeBins, ...
    valueDiffs, avgProbs, lateChoice, lateFixItem, lateFixTime, ...
    lateValueLeft, lateValueRight);
means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(corrProbs)
    means(j) = mean(corrProbs{j});
    stdev = std(corrProbs{j});
    confInts(j) = Get95PercConfidenceInterval(stdev, length(corrProbs{j}));
end

h2 = errorbar(1:length(timeBins), means, confInts, '--.', ...
    'MarkerSize', 25, 'Color', magenta, 'LineWidth', 3);

xlabel('Overall relative fixation left [ms]', 'FontSize', 24);
ylabel('Corrected p(choose left)','FontSize', 24);
legend([h1 h2], {'Early trials', 'Late trials'}, 'Location', 'SouthEast');
xlim([0 length(timeBins)+1]);

xlabels = cell(length(timeBins), 1);
for i=1:length(timeBins)
    xlabels{i,1} = sprintf('%d', timeBins(i));
end
set(gca, 'XTick', 1:length(timeBins), 'XTickLabel', xlabels);
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Probability of choosing first seen item vs. first fixation duration.

timeBins = 0:200:600;

[countChoseFirst, countTotal] = ProbChooseFirstVsTimeFirstFix( ...
    timeBins, earlyChoice, earlyFixItem, earlyFixTime);
means = zeros(length(timeBins), 1);
confInts = zeros(length(timeBins), 1);
for j=1:length(countChoseFirst)
    probChoseFirst = countChoseFirst(j) / countTotal(j);
    means(j,1) = probChoseFirst;
    stdev = sqrt(probChoseFirst .* (1-probChoseFirst));
    confInts(j,1) = Get95PercConfidenceInterval(stdev, countTotal(j));
end

figure; hold;
h1 = bar(1:length(timeBins), means, 'FaceColor', 0.8 + 0.2 * cyan, ...
    'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, 'LineWidth', 3);
errorbar(1:length(timeBins), means, confInts, '.', 'Color', ...
    0.7 * cyan, 'LineWidth', 3);

[countChoseFirst, countTotal] = ProbChooseFirstVsTimeFirstFix( ...
    timeBins, lateChoice, lateFixItem, lateFixTime);
means = zeros(length(timeBins), 1);
confInts = zeros(length(timeBins), 1);
for j=1:length(countChoseFirst)
    probChoseFirst = countChoseFirst(j) / countTotal(j);
    means(j,1) = probChoseFirst;
    stdev = sqrt(probChoseFirst .* (1-probChoseFirst));
    confInts(j,1) = Get95PercConfidenceInterval(stdev, countTotal(j));
end

h2 = errorbar(1:length(timeBins), means, confInts, '--.', ...
    'MarkerSize', 25, 'Color', magenta, 'LineWidth', 3);

xlabel('First fixation duration [ms]', 'FontSize', 24);
ylabel('P(first seen chosen)','FontSize', 24);
legend([h1 h2], {'Early trials', 'Late trials'}, 'Location', 'SouthEast');

xlabels = cell(length(timeBins), 1);
for i=1:length(timeBins)
    xlabels{i,1} = sprintf('%d', timeBins(i));
end
set(gca, 'XTick', 1:length(timeBins), 'XTickLabel', xlabels);
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Corrected probability of choosing first seen item vs. first fixation
% duration.

timeBins = 0:200:600;

avgProbs = GetAvgProbChooseFirstFixByValueDiff(valueDiffs, ...
    earlyValueLeft, earlyValueRight, earlyChoice, earlyFixItem, ...
    earlyFixTime);
corrProbs = CorrProbChooseFirstFixVsTimeFirstFix(timeBins, ...
    valueDiffs, avgProbs, earlyChoice, earlyFixItem, earlyFixTime, ...
    earlyValueLeft, earlyValueRight);
means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(corrProbs)
    means(j) = mean(corrProbs{j});
    stdev = std(corrProbs{j});
    confInts(j) = Get95PercConfidenceInterval(stdev, length(corrProbs{j}));
end

figure; hold;
h1 = bar(1:length(timeBins), means, 'FaceColor', 0.8 + 0.2 * cyan, ...
    'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, 'LineWidth', 3);
errorbar(1:length(timeBins), means, confInts, '.', 'Color', ...
    0.7 * cyan, 'LineWidth', 3);

avgProbs = GetAvgProbChooseFirstFixByValueDiff(valueDiffs, ...
    lateValueLeft, lateValueRight, lateChoice, lateFixItem, ...
    lateFixTime);
corrProbs = CorrProbChooseFirstFixVsTimeFirstFix(timeBins, ...
    valueDiffs, avgProbs, lateChoice, lateFixItem, lateFixTime, ...
    lateValueLeft, lateValueRight);
means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(corrProbs)
    means(j) = mean(corrProbs{j});
    stdev = std(corrProbs{j});
    confInts(j) = Get95PercConfidenceInterval(stdev, length(corrProbs{j}));
end

h2 = errorbar(1:length(timeBins), means, confInts, '--.', ...
    'MarkerSize', 25, 'Color', magenta, 'LineWidth', 3);

xlabel('First fixation duration [ms]', 'FontSize', 24);
ylabel('Corrected p(first seen chosen)','FontSize', 24);
legend([h1 h2], {'Early trials', 'Late trials'}, 'Location', 'SouthEast');

xlabels = cell(length(timeBins), 1);
for i=1:length(timeBins)
    xlabels{i,1} = sprintf('%d', timeBins(i));
end
set(gca, 'XTick', 1:length(timeBins), 'XTickLabel', xlabels);
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% RTs vs. value difference.

% Plot early trials reaction times.
reactionTimes = ReactionTimesByValueDiff(valueDiffs, earlyRt, ...
    earlyValueLeft, earlyValueRight);
means = zeros(length(valueDiffs),1);
confInt = zeros(length(valueDiffs),1);
for j=1:length(valueDiffs)
    means(j) = mean(reactionTimes{j});
    confInt(j) = Get95PercConfidenceInterval(std(reactionTimes{j}), ...
        length(reactionTimes{j}));
end
figure; hold;
shadedErrorBar(valueDiffs, means, confInt, {'-', 'Marker', '.', ...
    'MarkerSize', 20, 'Color', cyan, 'LineWidth', 3}, transparency);

% Plot late trials reaction times.
reactionTimes = ReactionTimesByValueDiff(valueDiffs, lateRt, ...
    lateValueLeft, lateValueRight);
means = zeros(length(valueDiffs),1);
confInt = zeros(length(valueDiffs),1);
for j=1:length(valueDiffs)
    means(j) = mean(reactionTimes{j});
    confInt(j) = Get95PercConfidenceInterval(std(reactionTimes{j}), ...
        length(reactionTimes{j}));
end
shadedErrorBar(valueDiffs, means, confInt, {'-', 'Marker', '.', ...
    'MarkerSize', 20, 'Color', magenta, 'LineWidth', 3}, transparency);
xlim(xLim);
ylim([0 3000]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('Reaction time [ms]','FontSize', 24);
legend('Early trials', 'Late trials');
set(gca, 'FontSize', 20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% RTs vs. value best minus value worst.

% Plot early trials reaction times.
reactionTimes = ReactionTimesVsValueBestMinusWorst( ...
    nonNegativeValueDiffs, earlyRt, earlyValueLeft, earlyValueRight);
means = zeros(length(nonNegativeValueDiffs),1);
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    means(j) = mean(reactionTimes{j});
    confInt(j) = Get95PercConfidenceInterval(std(reactionTimes{j}), ...
        length(reactionTimes{j}));
end

figure; hold;
h1 = bar(nonNegativeValueDiffs, means, 'FaceColor', ...
    0.8 + 0.2 * cyan, 'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, ...
    'LineWidth', 3);
errorbar(nonNegativeValueDiffs, means, confInt, '.', ...
    'Color', 0.7 * cyan, 'LineWidth', 3);

% Plot late trials reaction times.
reactionTimes = ReactionTimesVsValueBestMinusWorst( ...
    nonNegativeValueDiffs, lateRt, lateValueLeft, lateValueRight);
means = zeros(length(nonNegativeValueDiffs),1);
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    means(j) = mean(reactionTimes{j});
    confInt(j) = Get95PercConfidenceInterval(std(reactionTimes{j}), ...
        length(reactionTimes{j}));
end
h2 = errorbar(nonNegativeValueDiffs, means, confInt, '--.', ...
    'MarkerSize', 25, 'Color', magenta, 'LineWidth', 3);

xlabel('r_{closest} - r_{farthest}','FontSize', 24);
ylabel('Reaction time [ms]','FontSize', 24);
legend([h1 h2], {'Early trials', 'Late trials'}, 'Location', 'SouthEast');

xlim([-0.5 3.5]);
xlabels = cell(length(nonNegativeValueDiffs), 1);
for i=1:length(nonNegativeValueDiffs)
    xlabels{i,1} = sprintf('%d', nonNegativeValueDiffs(i));
end
set(gca, 'XTick', nonNegativeValueDiffs, 'XTickLabel', xlabels);
set(gca, 'FontSize', 20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Number of fixations vs. value best minus value worst.

% Plot early trials number of fixations.
numFixations = NumFixationsVsValueBestMinusWorst(nonNegativeValueDiffs, ...
    earlyFixItem, earlyValueLeft, earlyValueRight);
means = zeros(length(nonNegativeValueDiffs),1);
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    means(j) = mean(numFixations{j});
    confInt(j) = Get95PercConfidenceInterval(std(numFixations{j}), ...
        length(numFixations{j}));
end

figure; hold;
h1 = bar(nonNegativeValueDiffs, means, 'FaceColor', ...
    0.8 + 0.2 * cyan, 'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, ...
    'LineWidth', 3);
errorbar(nonNegativeValueDiffs, means, confInt, '.', ...
    'Color', 0.7 * cyan, 'LineWidth', 3);

% Plot late trials number of fixations.
numFixations = NumFixationsVsValueBestMinusWorst(nonNegativeValueDiffs, ...
    lateFixItem, lateValueLeft, lateValueRight);
means = zeros(length(nonNegativeValueDiffs),1);
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    means(j) = mean(numFixations{j});
    confInt(j) = Get95PercConfidenceInterval(std(numFixations{j}), ...
        length(numFixations{j}));
end
h2 = errorbar(nonNegativeValueDiffs, means, confInt, '--.', ...
    'MarkerSize', 25, 'Color', magenta, 'LineWidth', 3);

xlim([-0.5 3.5]);
xlabels = cell(length(nonNegativeValueDiffs), 1);
for i=1:length(nonNegativeValueDiffs)
    xlabels{i,1} = sprintf('%d', nonNegativeValueDiffs(i));
end
set(gca, 'XTick', nonNegativeValueDiffs, 'XTickLabel', xlabels);

xlabel('r_{closest} - r_{farthest}','FontSize', 24);
ylabel('Number of fixations','FontSize', 24);
legend([h1 h2], {'Early trials', 'Late trials'}, 'Location', 'SouthEast');
set(gca, 'FontSize', 20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Probability of error vs. value best minus value worst.

% Plot early trials probability of error.
[countError, countTotal] = ProbErrorGroupedVsValueBestMinusWorst( ...
    positiveValueDiffs, earlyChoice, earlyValueLeft, earlyValueRight);
figure; hold;
probError = countError ./ countTotal;
stdProbError = sqrt(probError .* (1-probError));
confInt = Get95PercConfidenceInterval(stdProbError, countTotal);
shadedErrorBar(positiveValueDiffs, probError, confInt, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan}, transparency);

% Plot late trials probability of error.
[countError, countTotal] = ProbErrorGroupedVsValueBestMinusWorst( ...
    positiveValueDiffs, lateChoice, lateValueLeft, lateValueRight);
probError = countError ./ countTotal;
stdProbError = sqrt(probError .* (1-probError));
confInt = Get95PercConfidenceInterval(stdProbError, countTotal);
shadedErrorBar(positiveValueDiffs, probError, confInt, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta}, transparency);

xlabel('r_{closest} - r_{farthest}','FontSize', 24);
ylabel('P(error)','FontSize', 24);
legend({'Early trials', 'Late trials'}, 'Location', 'NorthEast');

xlim([0.5 3.5]);
ylim([0 0.5]);
xlabels = cell(length(positiveValueDiffs), 1);
for i=1:length(positiveValueDiffs)
    xlabels{i,1} = sprintf('%d', positiveValueDiffs(i));
end
set(gca, 'XTick', positiveValueDiffs, 'XTickLabel', xlabels);
set(gca, 'FontSize', 20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% RT histograms.

% RT histogram for early trials reaction times.
reactionTimes = ReactionTimesByValueDiff(valueDiffs, earlyRt, ...
    earlyValueLeft, earlyValueRight);
rts = [];
for j=1:length(valueDiffs)
    rts = [rts; reactionTimes{j}];
end
figure; hold on;
[n,b] = hist(rts, 0:50:10000);
n = n / length(rts);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);

% RT histogram late trials reaction times.
reactionTimes = ReactionTimesByValueDiff(valueDiffs, lateRt, ...
    lateValueLeft, lateValueRight);
rts = [];
for j=1:length(valueDiffs)
    rts = [rts; reactionTimes{j}];
end
[n,b] = hist(rts, 0:50:10000);
n = n / length(rts);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);

xlabel('Reaction time [ms]','FontSize', 24);
ylabel('Relative frequency','FontSize', 24);
legend('Early trials', 'Late trials');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Fixation times vs. value best - value worst.

meanFirstFixTimes = zeros(length(nonNegativeValueDiffs),2);
confIntFirstFixTimes = zeros(length(nonNegativeValueDiffs),2);
meanSecondFixTimes = zeros(length(nonNegativeValueDiffs),2);
confIntSecondFixTimes = zeros(length(nonNegativeValueDiffs),2);
meanOtherFixTimes = zeros(length(nonNegativeValueDiffs),2);
confIntOtherFixTimes = zeros(length(nonNegativeValueDiffs),2);
meanLastFixTimes = zeros(length(nonNegativeValueDiffs),2);
confIntLastFixTimes = zeros(length(nonNegativeValueDiffs),2);

% Plot early trials fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueBestMinusWorst(nonNegativeValueDiffs, ...
    earlyFixItem, earlyFixTime, earlyValueLeft, earlyValueRight);
for i=1:length(nonNegativeValueDiffs)
    meanFirstFixTimes(i,1) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,1) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,1) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,1) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

% Plot late trials fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueBestMinusWorst(nonNegativeValueDiffs, ...
    lateFixItem, lateFixTime, lateValueLeft, lateValueRight);
for i=1:length(nonNegativeValueDiffs)
    meanFirstFixTimes(i,2) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,2) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,2) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,2) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

errorbar_groups(meanFirstFixTimes', confIntFirstFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    nonNegativeValueDiffs);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{closest} - r_{farthest}', 'FontSize', 24);
ylabel('First fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanSecondFixTimes', confIntSecondFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    nonNegativeValueDiffs);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{closest} - r_{farthest}', 'FontSize', 24);
ylabel('Second fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanOtherFixTimes', confIntOtherFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    nonNegativeValueDiffs);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{closest} - r_{farthest}', 'FontSize', 24);
ylabel('Other middle fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanLastFixTimes', confIntLastFixTimes', 'bar_colors', ...
    [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    nonNegativeValueDiffs);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{closest} - r_{farthest}', 'FontSize', 24);
ylabel('Last fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Fixation times vs. value fixated - value unfixated.

meanFirstFixTimes = zeros(length(valueDiffs),2);
confIntFirstFixTimes = zeros(length(valueDiffs),2);
meanSecondFixTimes = zeros(length(valueDiffs),2);
confIntSecondFixTimes = zeros(length(valueDiffs),2);
meanOtherFixTimes = zeros(length(valueDiffs),2);
confIntOtherFixTimes = zeros(length(valueDiffs),2);
meanLastFixTimes = zeros(length(valueDiffs),2);
confIntLastFixTimes = zeros(length(valueDiffs),2);

% Plot early trials fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, ...
    earlyFixItem, earlyFixTime, earlyValueLeft, earlyValueRight);
for i=1:length(valueDiffs)
    meanFirstFixTimes(i,1) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,1) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,1) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,1) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

% Plot late trials fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, ...
    lateFixItem, lateFixTime, lateValueLeft, lateValueRight);
for i=1:length(valueDiffs)
    meanFirstFixTimes(i,2) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,2) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,2) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,2) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

errorbar_groups(meanFirstFixTimes', confIntFirstFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    valueDiffs);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{fixated} - r_{unfixated}', 'FontSize', 24);
ylabel('First fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanSecondFixTimes', confIntSecondFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    valueDiffs);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{fixated} - r_{unfixated}', 'FontSize', 24);
ylabel('Second fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanOtherFixTimes', confIntOtherFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    valueDiffs);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{fixated} - r_{unfixated}', 'FontSize', 24);
ylabel('Other middle fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanLastFixTimes', confIntLastFixTimes', 'bar_colors', ...
    [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    valueDiffs);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{fixated} - r_{unfixated}', 'FontSize', 24);
ylabel('Last fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Fixation times vs. value fixated.

meanFirstFixTimes = zeros(length(values),2);
confIntFirstFixTimes = zeros(length(values),2);
meanSecondFixTimes = zeros(length(values),2);
confIntSecondFixTimes = zeros(length(values),2);
meanOtherFixTimes = zeros(length(values),2);
confIntOtherFixTimes = zeros(length(values),2);
meanLastFixTimes = zeros(length(values),2);
confIntLastFixTimes = zeros(length(values),2);

% Plot early trials fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFix(values, earlyFixItem, earlyFixTime, ...
    earlyValueLeft, earlyValueRight);
for i=1:length(values)
    meanFirstFixTimes(i,1) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,1) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,1) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,1) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

% Plot late trials fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFix(values, lateFixItem, lateFixTime, ...
    lateValueLeft, lateValueRight);
for i=1:length(values)
    meanFirstFixTimes(i,2) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,2) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,2) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,2) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

errorbar_groups(meanFirstFixTimes', confIntFirstFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    values);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{fixated}', 'FontSize', 24);
ylabel('First fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanSecondFixTimes', confIntSecondFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    values);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{fixated}', 'FontSize', 24);
ylabel('Second fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanOtherFixTimes', confIntOtherFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    values);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{fixated}', 'FontSize', 24);
ylabel('Other middle fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanLastFixTimes', confIntLastFixTimes', 'bar_colors', ...
    [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    valueDiffs);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{fixated}', 'FontSize', 24);
ylabel('Last fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Fixation times vs. value unfixated.

meanFirstFixTimes = zeros(length(values),2);
confIntFirstFixTimes = zeros(length(values),2);
meanSecondFixTimes = zeros(length(values),2);
confIntSecondFixTimes = zeros(length(values),2);
meanOtherFixTimes = zeros(length(values),2);
confIntOtherFixTimes = zeros(length(values),2);
meanLastFixTimes = zeros(length(values),2);
confIntLastFixTimes = zeros(length(values),2);

% Plot early trials fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueUnfix(values, earlyFixItem, earlyFixTime, ...
    earlyValueLeft, earlyValueRight);
for i=1:length(values)
    meanFirstFixTimes(i,1) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,1) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,1) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,1) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,1) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

% Plot late trials fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueUnfix(values, lateFixItem, lateFixTime, ...
    lateValueLeft, lateValueRight);
for i=1:length(values)
    meanFirstFixTimes(i,2) = mean(firstFixTimes{i,1});
    confIntFirstFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(firstFixTimes{i,1}), length(firstFixTimes{i,1}));
    meanSecondFixTimes(i,2) = mean(secondFixTimes{i,1});
    confIntSecondFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(secondFixTimes{i,1}), length(secondFixTimes{i,1}));
    meanOtherFixTimes(i,2) = mean(otherFixTimes{i,1});
    confIntOtherFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(otherFixTimes{i,1}), length(otherFixTimes{i,1}));
    meanLastFixTimes(i,2) = mean(lastFixTimes{i,1});
    confIntLastFixTimes(i,2) = Get95PercConfidenceInterval( ...
        std(lastFixTimes{i,1}), length(lastFixTimes{i,1}));
end

errorbar_groups(meanFirstFixTimes', confIntFirstFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    values);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{unfixated}', 'FontSize', 24);
ylabel('First fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanSecondFixTimes', confIntSecondFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    values);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{unfixated}', 'FontSize', 24);
ylabel('Second fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanOtherFixTimes', confIntOtherFixTimes', ...
    'bar_colors', [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    values);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
xlabel('r_{unfixated}', 'FontSize', 24);
ylabel('Other middle fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

errorbar_groups(meanLastFixTimes', confIntLastFixTimes', 'bar_colors', ...
    [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    valueDiffs);
legend('Early trials', 'Late trials');
xlabel('r_{unfixated}', 'FontSize', 24);
ylabel('Last fixation duration [ms]','FontSize', 24);
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Histogram of all types of fixations.

% Plot early trials fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, earlyFixItem, ...
    earlyFixTime, earlyValueLeft, earlyValueRight);

allEarlyFirstFixTime = [];
allEarlySecondFixTime = [];
allEarlyOtherFixTime = [];
allEarlyLastFixTime = [];
for j=1:length(valueDiffs)
    allEarlyFirstFixTime = [allEarlyFirstFixTime; firstFixTimes{j}];
    allEarlySecondFixTime = [allEarlySecondFixTime; secondFixTimes{j}];
    allEarlyOtherFixTime = [allEarlyOtherFixTime; otherFixTimes{j}];
    allEarlyLastFixTime = [allEarlyLastFixTime; lastFixTimes{j}];
end

% Plot late trials fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, lateFixItem, ...
    lateFixTime, lateValueLeft, lateValueRight);

allLateFirstFixTime = [];
allLateSecondFixTime = [];
allLateOtherFixTime = [];
allLateLastFixTime = [];
for j=1:length(valueDiffs)
    allLateFirstFixTime = [allLateFirstFixTime; firstFixTimes{j}];
    allLateSecondFixTime = [allLateSecondFixTime; secondFixTimes{j}];
    allLateOtherFixTime = [allLateOtherFixTime; otherFixTimes{j}];
    allLateLastFixTime = [allLateLastFixTime; lastFixTimes{j}];
end

figure; hold on;
[n,b] = hist(allEarlyFirstFixTime, 0:10:5000);
n = n / length(allEarlyFirstFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);
[n,b] = hist(allLateFirstFixTime, 0:10:5000);
n = n / length(allLateFirstFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);
xlim([0 1500]);
xlabel('First fixation time [ms]','FontSize', 24);
ylabel('Relative frequency','FontSize', 24);
legend('Early trials', 'Late trials');
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

figure; hold on;
[n,b] = hist(allEarlySecondFixTime, 0:10:5000);
n = n / length(allEarlySecondFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);
[n,b] = hist(allLateSecondFixTime, 0:10:5000);
n = n / length(allLateSecondFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);
xlim([0 1500]);
xlabel('Second fixation time [ms]','FontSize', 24);
ylabel('Relative frequency','FontSize', 24);
legend('Early trials', 'Late trials');
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

figure; hold on;
[n,b] = hist(allEarlyOtherFixTime, 0:10:5000);
n = n / length(allEarlyOtherFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);
[n,b] = hist(allLateOtherFixTime, 0:10:5000);
n = n / length(allLateOtherFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);
xlim([0 1500]);
xlabel('Other middle fixation time [ms]','FontSize', 24);
ylabel('Relative frequency','FontSize', 24);
legend('Early trials', 'Late trials');
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);

figure; hold on;
[n,b] = hist(allEarlyLastFixTime, 0:10:5000);
n = n / length(allEarlyLastFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);
[n,b] = hist(allLateLastFixTime, 0:10:5000);
n = n / length(allLateLastFixTime);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);
xlim([0 1500]);
xlabel('Last fixation time [ms]','FontSize', 24);
ylabel('Relative frequency','FontSize', 24);
legend('Early trials', 'Late trials');
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Histogram of last fixations grouped by choice.

[earlyLastFixTimesChoseFixed, earlyLastFixTimesChoseUnfixed] = ...
    LastFixationDurationGroupedByChoice(earlyChoice, earlyFixItem, ...
    earlyFixTime);

% Plot simulated fixation times.
[lateLastFixTimesChoseFixed, lateLastFixTimesChoseUnfixed] = ...
    LastFixationDurationGroupedByChoice(lateChoice, lateFixItem, ...
    lateFixTime);

figure; hold on;
[n,b] = hist(earlyLastFixTimesChoseFixed, 0:10:1500);
n = n / sum(n);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);

[n,b] = hist(earlyLastFixTimesChoseUnfixed, 0:10:1500);
n = n / sum(n);
plot(b, n, '--', 'Marker', 's', 'MarkerSize', 10, 'Color', cyan);

[n,b] = hist(lateLastFixTimesChoseFixed, 0:10:1500);
n = n / sum(n);
plot(b, n, '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);

[n,b] = hist(lateLastFixTimesChoseUnfixed, 0:10:1500);
n = n / sum(n);
plot(b, n, '--', 'Marker', 's', 'MarkerSize', 10, 'Color', magenta);

xlim([0 1200]);
xlabel('Last fixation time [ms]','FontSize', 24);
ylabel('Relative frequency','FontSize', 24);
legend('Early chose fixed', 'Early chose unfixed', 'Late chose fixed', ...
    'Late chose unfixed');
set(gca,'FontSize',20);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Probability of error vs value best minus value worst, grouped by choice.

% Early trials.
[countErrorChoseFixed, countTotalChoseFixed, countErrorChoseUnfixed, ...
    countTotalChoseUnfixed] = ProbErrorGroupedByChoice( ...
    nonNegativeValueDiffs, earlyChoice, earlyFixItem, earlyValueLeft, ...
    earlyValueRight);

figure; hold on;
probErrorChoseFixed = countErrorChoseFixed ./ countTotalChoseFixed;
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    standardDev = sqrt(probErrorChoseFixed(j) .* ...
        (1-probErrorChoseFixed(j)));
    confInt(j) = Get95PercConfidenceInterval(standardDev, ...
        countTotalChoseFixed(j));
end
errorbar(nonNegativeValueDiffs(2:end), probErrorChoseFixed(2:end), ...
    confInt(2:end), '-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan);

probErrorChoseUnfixed = countErrorChoseUnfixed ./ countTotalChoseUnfixed;
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    standardDev = sqrt(probErrorChoseUnfixed(j) .* ...
        (1-probErrorChoseUnfixed(j)));
    confInt(j) = Get95PercConfidenceInterval(standardDev, ...
        countTotalChoseUnfixed(j));
end
errorbar(nonNegativeValueDiffs(2:end), probErrorChoseUnfixed(2:end), ...
    confInt(2:end), '--', 'Marker', 's', 'MarkerSize', 10, 'Color', cyan);

% Late trials.
[countErrorChoseFixed, countTotalChoseFixed, countErrorChoseUnfixed, ...
    countTotalChoseUnfixed] = ProbErrorGroupedByChoice( ...
    nonNegativeValueDiffs, lateChoice, lateFixItem, lateValueLeft, ...
    lateValueRight);

probErrorChoseFixed = countErrorChoseFixed ./ countTotalChoseFixed;
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    standardDev = sqrt(probErrorChoseFixed(j) .* ...
        (1-probErrorChoseFixed(j)));
    confInt(j) = Get95PercConfidenceInterval(standardDev, ...
        countTotalChoseFixed(j));
end
errorbar(nonNegativeValueDiffs(2:end), probErrorChoseFixed(2:end), ...
    confInt(2:end), '-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta);

probErrorChoseUnfixed = countErrorChoseUnfixed ./ countTotalChoseUnfixed;
confInt = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    standardDev = sqrt(probErrorChoseUnfixed(j) .* ...
        (1-probErrorChoseUnfixed(j)));
    confInt(j) = Get95PercConfidenceInterval(standardDev, ...
        countTotalChoseUnfixed(j));
end
errorbar(nonNegativeValueDiffs(2:end), probErrorChoseUnfixed(2:end), ...
    confInt(2:end), '--', 'Marker', 's', 'MarkerSize', 10, 'Color', ...
    magenta);

xlabel('r_{closest} - r_{farthest}','FontSize', 24);
ylabel('P(error)','FontSize', 24);
legend('Early chose fixed', 'Early chose unfixed', 'Late chose fixed', ...
    'Late chose unfixed');
set(gca,'FontSize',20);
xlim([0.5 3.5]);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Number of fixations by value difference, grouped by whether last fixated
% item was chosen.

% Early trials.
[earlyNumFixationsChoseFixed, earlyNumFixationsChoseUnfixed] = ...
    NumFixationsByChoiceAndValueDiff(nonNegativeValueDiffs, ...
    earlyChoice, earlyFixItem, earlyFixTime, earlyValueLeft, earlyValueRight);

% Late trials.
[lateNumFixationsChoseFixed, lateNumFixationsChoseUnfixed] = ...
    NumFixationsByChoiceAndValueDiff(nonNegativeValueDiffs, ...
    lateChoice, lateFixItem, lateFixTime, lateValueLeft, ...
    lateValueRight);

means = zeros(length(nonNegativeValueDiffs),4);
for j=1:length(nonNegativeValueDiffs)
    means(j,1) = mean(earlyNumFixationsChoseFixed{j});
    means(j,2) = mean(earlyNumFixationsChoseUnfixed{j});
    means(j,3) = mean(lateNumFixationsChoseFixed{j});
    means(j,4) = mean(lateNumFixationsChoseUnfixed{j});
end

confInts = zeros(length(nonNegativeValueDiffs),4);
for j=1:length(nonNegativeValueDiffs)
    confInts(j,1) = Get95PercConfidenceInterval( ...
        std(earlyNumFixationsChoseFixed{j}), ...
        length(earlyNumFixationsChoseFixed{j}));
    confInts(j,2) = Get95PercConfidenceInterval( ...
        std(earlyNumFixationsChoseUnfixed{j}), ...
        length(earlyNumFixationsChoseUnfixed{j}));
    confInts(j,3) = Get95PercConfidenceInterval( ...
        std(lateNumFixationsChoseFixed{j}), ...
        length(lateNumFixationsChoseFixed{j}));
    confInts(j,4) = Get95PercConfidenceInterval( ...
        std(lateNumFixationsChoseUnfixed{j}), ...
        length(lateNumFixationsChoseUnfixed{j}));
end

errorbar_groups(means', confInts', 'bar_colors', ...
    [0.2 + 0.8 * cyan; 0.8 + 0.2 * cyan; 0.2 + 0.8 * magenta; ...
    0.8 + 0.2 * magenta;], 'bar_names', nonNegativeValueDiffs);
xlabel('r_{closest} - r_{farthest}', 'FontSize', 24);
ylabel('Number of fixations','FontSize', 24);
legend('Early chose last fix', 'Early chose other', ...
    'Late chose last fix', 'Late chose other', 'Location', 'SouthEast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Number of trials by value difference, grouped by whether last fixated
% item was chosen.

% Early trials.
[earlyNumTrialsChoseFixed, earlyNumTrialsChoseUnfixed] = ...
    TrialCountByChoiceAndValueDiff(nonNegativeValueDiffs, ...
    earlyChoice, earlyFixItem, earlyValueLeft, earlyValueRight);

% Late trials.
[lateNumTrialsChoseFixed, lateNumTrialsChoseUnfixed] = ...
    TrialCountByChoiceAndValueDiff(nonNegativeValueDiffs, ...
    lateChoice, lateFixItem, lateValueLeft, lateValueRight);

values = [earlyNumTrialsChoseFixed, earlyNumTrialsChoseUnfixed, ...
    lateNumTrialsChoseFixed, lateNumTrialsChoseUnfixed];

errorbar_groups(values', zeros(4,4), 'bar_colors', ...
    [0.2 + 0.8 * cyan; 0.8 + 0.2 * cyan; 0.2 + 0.8 * magenta; ...
    0.8 + 0.2 * magenta;], 'bar_names', nonNegativeValueDiffs);
xlabel('r_{closest} - r_{farthest}', 'FontSize', 24);
ylabel('Number of trials','FontSize', 24);
legend('Early chose last fix', 'Early chose other', ...
    'Late chose last fix', 'Late chose other', 'Location', 'NorthEast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Plot bar graph of fixation times for each type of fixation.

means = zeros(4,2);
confInts = zeros(4,2);

% Plot early trials fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, earlyFixItem, ...
    earlyFixTime, earlyValueLeft, earlyValueRight);
dataFirstFixTimes = [];
dataSecondFixTimes = [];
dataOtherFixTimes = [];
dataLastFixTimes = [];
for j=1:length(valueDiffs)
    dataFirstFixTimes = [dataFirstFixTimes; firstFixTimes{j,1}];
    dataSecondFixTimes = [dataSecondFixTimes; secondFixTimes{j,1}];
    dataOtherFixTimes = [dataOtherFixTimes; otherFixTimes{j,1}];
    dataLastFixTimes = [dataLastFixTimes; lastFixTimes{j,1}];
end
means(1,1) = mean(dataFirstFixTimes);
means(2,1) = mean(dataSecondFixTimes);
means(3,1) = mean(dataOtherFixTimes);
means(4,1) = mean(dataLastFixTimes);
confInts(1,1) = Get95PercConfidenceInterval(std(dataFirstFixTimes), ...
    length(dataFirstFixTimes));
confInts(2,1) = Get95PercConfidenceInterval(std(dataSecondFixTimes), ...
    length(dataSecondFixTimes));
confInts(3,1) = Get95PercConfidenceInterval(std(dataOtherFixTimes), ...
    length(dataOtherFixTimes));
confInts(4,1) = Get95PercConfidenceInterval(std(dataLastFixTimes), ...
    length(dataLastFixTimes));

% Plot late trials fixation times.
[firstFixTimes, secondFixTimes, otherFixTimes, lastFixTimes] = ...
    FixationDurationVsValueFixMinusUnfix(valueDiffs, lateFixItem, ...
    lateFixTime, lateValueLeft, lateValueRight);
lateFirstFixTimes = [];
lateSecondFixTimes = [];
lateOtherFixTimes = [];
lateLastFixTimes = [];
for j=1:length(valueDiffs)
    lateFirstFixTimes = [lateFirstFixTimes; firstFixTimes{j,1}];
    lateSecondFixTimes = [lateSecondFixTimes; secondFixTimes{j,1}];
    lateOtherFixTimes = [lateOtherFixTimes; otherFixTimes{j,1}];
    lateLastFixTimes = [lateLastFixTimes; lastFixTimes{j,1}];
end
means(1,2) = mean(lateFirstFixTimes);
means(2,2) = mean(lateSecondFixTimes);
means(3,2) = mean(lateOtherFixTimes);
means(4,2) = mean(lateLastFixTimes);
confInts(1,2) = Get95PercConfidenceInterval(std(lateFirstFixTimes), ...
    length(lateFirstFixTimes));
confInts(2,2) = Get95PercConfidenceInterval( ...
    std(lateSecondFixTimes), length(lateSecondFixTimes));
confInts(3,2) = Get95PercConfidenceInterval( ...
    std(lateOtherFixTimes), length(lateOtherFixTimes));
confInts(4,2) = Get95PercConfidenceInterval(std(lateLastFixTimes), ...
    length(lateLastFixTimes));

errorbar_groups(means', confInts', 'bar_colors', ...
    [0.8 + 0.2 * cyan; 0.8 + 0.2 * magenta], 'bar_names', ...
    1:3);
set(gca, 'XTick', 1:8, 'XTickLabel', {'1st','','2nd','', ...
    'Other', '', 'Last', ''});
ylabel('Fixation duration [ms]','FontSize', 24);
legend('Early trials', 'Late trials');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Probability that last fixation is to chosen item vs. value of last
% fixated item minus other value.

% Plot early trials.
[countLastFixToChosen, countTotal] = ...
    ProbLastFixIsToChosenItemVsValueLastSeenMinusOther(valueDiffs, ...
    earlyFixItem, earlyChoice, earlyValueLeft, earlyValueRight);
probLastFixToChosen = countLastFixToChosen ./ countTotal;
stdProbLastFixToChosen = sqrt(probLastFixToChosen .* ...
    (1-probLastFixToChosen));
confInt = Get95PercConfidenceInterval(stdProbLastFixToChosen, ...
    countTotal);
figure; hold;
shadedErrorBar(valueDiffs, probLastFixToChosen, confInt, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

% Size of bias effect.
bias = probLastFixToChosen(4) - 0.5;

% Plot late trials.
[countLastFixToChosen, countTotal] = ...
    ProbLastFixIsToChosenItemVsValueLastSeenMinusOther(valueDiffs, ...
    lateFixItem, lateChoice, lateValueLeft, lateValueRight);
probLastFixToChosen = countLastFixToChosen ./ countTotal;
stdProbLastFixToChosen = sqrt(probLastFixToChosen .* ...
    (1-probLastFixToChosen));
confInt = Get95PercConfidenceInterval(stdProbLastFixToChosen, countTotal);
shadedErrorBar(valueDiffs, probLastFixToChosen, confInt, ...
    {'-', 'Marker', '.', 'MarkerSize', 20, 'Color', magenta, ...
    'LineWidth', 3}, transparency);

line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5], ...
    'LineWidth', 2);
line([0 0],[0 1],'LineStyle','--','Color',[0.5 0.5 0.5], 'LineWidth', 2);
xlim(xLim);
ylim([0 1]);
xlabel('r_{last seen item} - r_{other item}','FontSize', 24);
ylabel('P(choose last fixated item)','FontSize', 24);
legend('Early trials', 'Late trials', 'Location', 'SouthEast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Time advantage of item B over item A (last fixation was to A) before
% last fixation vs last fixation duration.

timeBinLimits = 0:300:900;

% Plot early trials.
timeAdvantage = TimeAdvantageUnfixItemVsLastFixTime(timeBinLimits, ...
    earlyFixItem, earlyFixTime, earlyChoice);
means = zeros(length(timeBinLimits), 1);
confInts = zeros(length(timeBinLimits), 1);
for i=1:length(timeBinLimits)
   means(i,1) = mean(timeAdvantage{i});
   confInts(i,1) = Get95PercConfidenceInterval(std(timeAdvantage{i}), ...
       length(timeAdvantage{i}));
end
xlabels = cell(length(timeBinLimits),1);
for i=1:length(timeBinLimits)
    xlabels{i,1} = sprintf('%d', timeBinLimits(i));
end

figure; hold on;
h1 = bar(means, 'FaceColor', 0.8 + 0.2 * cyan, 'EdgeColor', ...
    0.7 * cyan, 'BarWidth', 0.4, 'LineWidth', 3);
errorbar(means, confInts, '.', 'Color', 0.7 * cyan, 'LineWidth', 3);
set(gca, 'XTick', 1:length(timeBinLimits), 'XTickLabel', xlabels);

% Plot late trials.
timeAdvantage = TimeAdvantageUnfixItemVsLastFixTime(timeBinLimits, ...
    lateFixItem, lateFixTime, lateChoice);
means = zeros(length(timeBinLimits), 1);
confInts = zeros(length(timeBinLimits), 1);
for i=1:length(timeBinLimits)
   means(i,1) = mean(timeAdvantage{i});
   confInts(i,1) = Get95PercConfidenceInterval(std(timeAdvantage{i}), ...
       length(timeAdvantage{i}));
end

h2 = errorbar(means, confInts, '--.', 'MarkerSize', 25, ...
    'Color', magenta, 'LineWidth', 3);

xlabel('Last fixation duration [ms]','FontSize', 24);
ylabel('Relative fixation non-chosen item [ms]','FontSize', 24);
legend([h1 h2], {'Early trials', 'Late trials'}, 'Location', 'SouthWest');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


close all
