function [fixTime, fixLocX, fixLocY, fixItem, trialDuration, ...
    transitionTime, mergedBlankFixations] = ReadEyetrackingData( ...
    parcode, sessn)

% This function classifies saccades as blank fixations. In addition,
% consecutive fixations on the same item are grouped together into the same
% fixation. Only the x and y coordinates of the first fixation in that
% series are stored.

% Load file with experimental data.
files = dir(sprintf('expdata/expdata_%s_s%d_*.mat', parcode, sessn));
load(sprintf('expdata/%s', files(1).name));

% Open file with eye tracking data.
fileName = ['eyetracking_data/' parcode '_s' int2str(sessn) '.asc']; 
fileId = fopen(fileName, 'r');

% Initialize eye tracking data arrays.
fixTime = cell(data.numBlocks, data.numTestTrials);
fixLocX = cell(data.numBlocks, data.numTestTrials);
fixLocY = cell(data.numBlocks, data.numTestTrials);
fixItem = cell(data.numBlocks, data.numTestTrials);
trialDuration = zeros(data.numBlocks, data.numTestTrials);
transitionTime = zeros(data.numBlocks, data.numTestTrials);

mergedBlankFixations = [];

% Parse data from eye tracking file.
for numBlock = 1:data.numBlocks
    for numTrial = 1:data.numTestTrials

        % Find the start of this trial's data.
        trialString = sprintf('TRIALID T_%d_%d', numBlock, numTrial);
        while true
            line = fgetl(fileId);
            if strfind(line, trialString)
                break;
            end
        end

        % Find the sync time message for this trial.
        while true
            line = fgetl(fileId);
            if strfind(line, 'SYNCTIME')
                res = sscanf(line,'%*s %d', 1);
                firstFixStartTime = res(1);
                break;
            end
        end

        % Get all the fixation and saccade events for this trial.
        % Note: saccades are classified the same as fixations on blank
        % locations (i.e. not left, right or center), and have fixLocX and
        % fixLocY set to -1.
        isFirstFix = true;
        numEvent = 1;
        decisionTime = 0;
        totalTime = 0;
        while true
            line = fgetl(fileId);
            if strfind(line, 'EFIX')
                res = sscanf(line,'%*s %*s %*d %d %d %f %f', 4);
                fixEndTime = res(1);
                time = res(2);
                xPos = res(3);
                yPos = res(4);
                
                item = GetItem(xPos, yPos, data);

                % Initial central fixation.
                if isFirstFix && item == 3
                    isFirstFix = false;
                    time = fixEndTime - firstFixStartTime;
                    fixTime{numBlock,numTrial}(numEvent) = time;
                    fixLocX{numBlock,numTrial}(numEvent) = xPos;
                    fixLocY{numBlock,numTrial}(numEvent) = yPos;
                    fixItem{numBlock,numTrial}(numEvent) = item;
                    numEvent = numEvent + 1;
                elseif numEvent > 1 && ...
                    item == fixItem{numBlock,numTrial}(numEvent-1)
                    fixTime{numBlock,numTrial}(numEvent-1) = ...
                        fixTime{numBlock,numTrial}(numEvent-1) + time;
                else
                    fixTime{numBlock,numTrial}(numEvent) = time;
                    fixLocX{numBlock,numTrial}(numEvent) = xPos;
                    fixLocY{numBlock,numTrial}(numEvent) = yPos;
                    fixItem{numBlock,numTrial}(numEvent) = item;
                    numEvent = numEvent + 1;
                end
                totalTime = totalTime + time;
                if item == 1 || item == 2
                    decisionTime = decisionTime + time;
                end
            elseif strfind(line, 'ESACC')
                res = sscanf(line,'%*s %*s %*d %*d %d', 1);
                time = res(1);

                if numEvent > 1 && ...
                        fixItem{numBlock,numTrial}(numEvent-1) == 0
                    fixTime{numBlock,numTrial}(numEvent-1) = ...
                        fixTime{numBlock,numTrial}(numEvent-1) + time;
                else
                    fixTime{numBlock,numTrial}(numEvent) = time;
                    fixLocX{numBlock,numTrial}(numEvent) = -1;
                    fixLocY{numBlock,numTrial}(numEvent) = -1;
                    fixItem{numBlock,numTrial}(numEvent) = 0;

                    numEvent = numEvent + 1;
                end
                totalTime = totalTime + time;
            elseif any(strfind(line, 'END'))
                trialDuration(numBlock,numTrial) = totalTime;
                transitionTime(numBlock,numTrial) = ...
                    totalTime - decisionTime;
                break;
            end
        end

        % Transform a middle or blank fixation into an item
        % fixation if it is sandwiched between 2 fixations to the
        % same item.
        trialFixItems = [];
        trialFixTimes = [];
        fix = 1;
        for i = 1:length(fixItem{numBlock,numTrial})
            % Item fixations.
            if fixItem{numBlock,numTrial}(i) == 1 ...
                    || fixItem{numBlock,numTrial}(i) == 2
                if fix > 1 && trialFixItems(fix-1) == ...
                        fixItem{numBlock,numTrial}(i)
                    trialFixTimes(fix-1) = ...
                        trialFixTimes(fix-1) + ...
                        fixTime{numBlock,numTrial}(i);
                else
                    trialFixItems(fix) = ...
                        fixItem{numBlock,numTrial}(i);
                    trialFixTimes(fix) = ...
                        fixTime{numBlock,numTrial}(i);
                    fix = fix + 1;
                end
            % Blank or middle fixations, sandwiched.
            elseif fix > 1 ...
                    && i < length(fixItem{numBlock,numTrial}) ...
                    && (trialFixItems(fix-1) == 1 ...
                    || trialFixItems(fix-1) == 2) ... 
                    && trialFixItems(fix-1) == ...
                    fixItem{numBlock,numTrial}(i+1)
                trialFixTimes(fix-1) = trialFixTimes(fix-1) + ...
                    fixTime{numBlock,numTrial}(i);
                mergedBlankFixations = [mergedBlankFixations; ...
                    fixTime{numBlock,numTrial}(i)];
            % Blank or middle fixations, non-sandwiched.
            else
                trialFixItems(fix) = fixItem{numBlock,numTrial}(i);
                trialFixTimes(fix) = fixTime{numBlock,numTrial}(i);
                fix = fix + 1;
            end
        end
        fixTime{numBlock,numTrial} = trialFixTimes;
        fixItem{numBlock,numTrial} = trialFixItems;
    end
end
end

% Classify coordinates of fixations into item numbers that were attended.
function item = GetItem(xPos, yPos, data)
    buffer = 1.8 * data.barLength;
    if xPos >= data.screenLeft(1) - buffer && ...
            xPos <= data.screenLeft(1) + buffer && ...
            yPos >= data.screenLeft(2) - buffer && ...
            yPos <= data.screenLeft(2) + buffer
        item = 1;  % 1 for LEFT
    elseif xPos >= data.screenRight(1) - buffer && ...
            xPos <= data.screenRight(1) + buffer && ...
            yPos >= data.screenRight(2) - buffer && ...
            yPos <= data.screenRight(2) + buffer
        item = 2;  % 2 for RIGHT
    elseif xPos >= data.screenCenter(1) - buffer && ...
            xPos <= data.screenCenter(1) + buffer && ...
            yPos >= data.screenCenter(2) - buffer && ...
            yPos <= data.screenCenter(2) + buffer
        item = 3;  % 3 for CENTER
    else
        item = 0;  % 0 for BLANK
    end
end
