close all
clear
clc

colors = distinguishable_colors(30);
cyan = colors(11,:);
pink = colors(5,:);
green = colors(13,:);
purple = colors(16,:);
magenta = colors(15,:);
yellow = colors(6,:);
alpha = 0.5;

transparency = 0;

valueDiffs = -15:5:15;
nonNegativeValueDiffs = 0:5:15;

plotValueDiffs = -3:1:3;
plotNonNegativeValueDiffs = 0:1:3;
xLim = [-3.5 3.5];

sessions = 1:1;
parcodes = ['som'; 'bem'; 'sim'; 'nao'; 'veu'; 'via'; 'leu'; 'ora'; ...
    'bar'; 'aro'; 'asa'; 'eco'; 'ego'; 'ano'; 'era'; 'ave'; 'cao'; ...
    'cem'; 'boi'; 'dom'];

oldParcodes = ['mar'; 'sol'; 'lua'; 'rio'; 'cor'; 'sal'; 'mel'; 'dia'; ...
    'ceu'; 'lei'; 'fio'; 'sul'; 'vai'; 'pai'; 'reu'; 'cai'; 'rei'; ...
    'dor'; 'giz'; 'mae'; 'tia'; 'pia'; 'fui'; 'gol'; 'gel'];
oldSessions = 1:4;

plotsFileBaseName = 'plots';
clocknum = clock;
dateString = [];
for i=1:5
    dateString = [dateString '_' num2str(clocknum(i))];
end
plotsFileName = [plotsFileBaseName dateString];


%% Get mean and SD for delay in showing biased item.

delays = [];
for i=1:length(parcodes)
    for s=sessions
        delays = [delays; GetFinalStaircaseDelay(parcodes(i,:), s)];
    end
end
disp(mean(delays));
disp(std(delays));


%% Check if manipulation is working.

% Check in how many trials the biased item was fixated longer.

aggregCountBiasedLongFix = 0;
aggregCountBiasedShortFix = 0;
aggregCountInvalid = 0;

for i=1:length(parcodes)
    subjTrialDuration = cell(length(valueDiffs), 1);

    for s=sessions
       [countBiasedLongFix, countBiasedShortFix, countInvalid] = ...
           CountTrialsPerBiasedAndFixTime(parcodes(i,:), s);
       
       aggregCountBiasedLongFix = aggregCountBiasedLongFix + ...
           countBiasedLongFix; 
       aggregCountBiasedShortFix = aggregCountBiasedShortFix + ...
           countBiasedShortFix;
       aggregCountInvalid = aggregCountInvalid + countInvalid;
    end
end

total = aggregCountBiasedLongFix + aggregCountBiasedShortFix + ...
    aggregCountInvalid;
percBiasedLongFix = (aggregCountBiasedLongFix / total) * 100;

percBiasedLongFix2 = (aggregCountBiasedLongFix / ...
    (aggregCountBiasedLongFix + aggregCountBiasedShortFix)) * 100;

% Check in how many trials the total fixation time allowed was exceeded.

countTotal = 0;
countExceeded = 0;
subjPercExceeded = [];

for i=1:length(parcodes)
    subjTrialDuration = cell(length(valueDiffs), 1);
    subjCountTotal = 0;
    subjCountExceeded = 0;
    for s=sessions
        files = dir(sprintf('expdata/expdata_%s_s%d_*.mat', ...
            parcodes(i,:), s));
        load(sprintf('expdata/%s', files(1).name));
        
        for numBlock = 1:data.numBlocks
            for numTrial = 1:data.numTestTrials
                subjCountTotal = subjCountTotal + 1;
                if data.testTotalFixTime{numBlock}(numTrial) >= 3
                    subjCountExceeded = subjCountExceeded + 1;
                end
            end
        end
    
    end
    subjPercExceeded = [subjPercExceeded;  ...
        subjCountExceeded / subjCountTotal];
    countTotal = countTotal + subjCountTotal;
    countExceeded = countExceeded + subjCountExceeded;
end
percExceeded = (countExceeded / countTotal) * 100;

figure;
text(0, 0, sprintf('Number of trials:\n'), 'FontSize', 28);
text(0, 1, sprintf('Time exceeded (discarded):  %d (%.1f%%)\n', ...
    countExceeded, percExceeded), 'FontSize', 28);
text(0, 2, sprintf('Biased fixated longer (effective):  %d (%.1f%%)\n', ...
    aggregCountBiasedLongFix, percBiasedLongFix), 'FontSize', 28);
text(0, 3, sprintf('Effective out of undiscarded:  %d (%.1f%%)\n', ...
    aggregCountBiasedLongFix, percBiasedLongFix2), 'FontSize', 28);
text(0, 4, sprintf('Total:  %d\n', countTotal), 'FontSize', 28);
axis ij
axis off
ylim([-1 5]);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Generate psychometric choice curves.

aggregCountDecisions = zeros(length(valueDiffs),1);
aggregCountLeftChosen = zeros(length(valueDiffs),1);

figure; hold all;

for i=1:length(parcodes)
    subjCountDecisions = zeros(length(valueDiffs),1);
    subjCountLeftChosen = zeros(length(valueDiffs),1);
    
    for s=sessions
        [countDecisions, countLeftChosen] = Choices(valueDiffs, ...
            parcodes(i,:), s);
        subjCountDecisions = subjCountDecisions + countDecisions;
        subjCountLeftChosen = subjCountLeftChosen + countLeftChosen;
    end
    
    subjProbLeftChosen = subjCountLeftChosen ./ subjCountDecisions;

    % Plot individual choice curve.
    c = alpha + (1-alpha) * colors(i,:);
    plot(plotValueDiffs, subjProbLeftChosen, '--.', 'MarkerSize', 25, ...
        'Color', c);

    aggregCountDecisions = aggregCountDecisions + subjCountDecisions;
    aggregCountLeftChosen = aggregCountLeftChosen + subjCountLeftChosen;
end

% Plot aggregated choice curve.
aggregProbLeftChosen = aggregCountLeftChosen ./ aggregCountDecisions;
stdProbLeftChosen = sqrt(aggregProbLeftChosen .* ...
    (1-aggregProbLeftChosen) ./ aggregCountDecisions);
errorbar(plotValueDiffs, aggregProbLeftChosen, stdProbLeftChosen, ...
    'LineStyle', '-', 'Marker', '.', 'MarkerSize', 25, 'Color', 'k', ...
    'LineWidth', 2);
line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5]);
xlim(xLim);
ylim([0 1]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Choice curves grouped by biased item.

aggregCountDecisionsLeftBiased = zeros(length(valueDiffs),1);
aggregCountLeftChosenLeftBiased = zeros(length(valueDiffs),1);
aggregCountDecisionsRightBiased = zeros(length(valueDiffs),1);
aggregCountLeftChosenRightBiased = zeros(length(valueDiffs),1);

for i=1:length(parcodes)
    subjCountDecisionsLeftBiased = zeros(length(valueDiffs),1);
    subjCountLeftChosenLeftBiased = zeros(length(valueDiffs),1);
    subjCountDecisionsRightBiased = zeros(length(valueDiffs),1);
    subjCountLeftChosenRightBiased = zeros(length(valueDiffs),1);
    
    for s=sessions
        [countDecisionsLeftBiased, countLeftChosenLeftBiased, ...
            countDecisionsRightBiased, countLeftChosenRightBiased] = ...
            ChoicesGroupedByBiased(valueDiffs, parcodes(i,:), s);
        subjCountDecisionsLeftBiased = subjCountDecisionsLeftBiased + ...
            countDecisionsLeftBiased;
        subjCountLeftChosenLeftBiased = subjCountLeftChosenLeftBiased + ...
            countLeftChosenLeftBiased;
        subjCountDecisionsRightBiased = subjCountDecisionsRightBiased + ...
            countDecisionsRightBiased;
        subjCountLeftChosenRightBiased = ...
            subjCountLeftChosenRightBiased + countLeftChosenRightBiased;
    end
    
    subjProbLeftChosenLeftBiased = subjCountLeftChosenLeftBiased ./ ...
        subjCountDecisionsLeftBiased;
    subjProbLeftChosenRightBiased = subjCountLeftChosenRightBiased ./ ...
        subjCountDecisionsRightBiased;

    aggregCountDecisionsLeftBiased = aggregCountDecisionsLeftBiased + ...
        subjCountDecisionsLeftBiased;
    aggregCountLeftChosenLeftBiased = aggregCountLeftChosenLeftBiased + ...
        subjCountLeftChosenLeftBiased;
    aggregCountDecisionsRightBiased = aggregCountDecisionsRightBiased + ...
        subjCountDecisionsRightBiased;
    aggregCountLeftChosenRightBiased = ...
        aggregCountLeftChosenRightBiased + subjCountLeftChosenRightBiased;
end

% Plot aggregated choice curves.
figure; hold;
aggregProbLeftChosenLeftBiased = aggregCountLeftChosenLeftBiased ./ ...
    aggregCountDecisionsLeftBiased;
stdProbLeftChosenLeftBiased = sqrt(aggregProbLeftChosenLeftBiased .* ...
    (1-aggregProbLeftChosenLeftBiased));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenLeftBiased, ...
    aggregCountDecisionsLeftBiased);
shadedErrorBar(plotValueDiffs, aggregProbLeftChosenLeftBiased, confInt, ...
    {'-', 'Marker', '.', 'MarkerSize', 24, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

aggregProbLeftChosenRightBiased = aggregCountLeftChosenRightBiased ./ ...
    aggregCountDecisionsRightBiased;
stdProbLeftChosenRightBiased = sqrt(aggregProbLeftChosenRightBiased .* ...
    (1-aggregProbLeftChosenRightBiased));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenRightBiased, ...
    aggregCountDecisionsRightBiased);
shadedErrorBar(plotValueDiffs, aggregProbLeftChosenRightBiased, ...
    confInt, {'-', 'Marker', '.', 'MarkerSize', 24, 'Color', yellow, ...
    'LineWidth', 3}, transparency);

line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5], ...
    'LineWidth', 2);
xlim(xLim);
ylim([0 1]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
legend('Left bias-target', 'Right bias-target', 'Location', 'Southeast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);

% Size of bias effect.
bias = aggregProbLeftChosenLeftBiased(4) - ...
    aggregProbLeftChosenRightBiased(4);

% Calculate chi-squared test on the size of the bias effect.
% Observed data
n1 = aggregCountLeftChosenLeftBiased(4);
N1 = aggregCountDecisionsLeftBiased(4);
n2 = aggregCountLeftChosenRightBiased(4);
N2 = aggregCountDecisionsRightBiased(4);
% Pooled estimate of proportion
p0 = (n1+n2) / (N1+N2)
% Expected counts under H0 (null hypothesis)
n10 = N1 * p0;
n20 = N2 * p0;
% Chi-square test, by hand
observed = [n1 N1-n1 n2 N2-n2];
expected = [n10 N1-n10 n20 N2-n20];
chi2stat = sum((observed-expected).^2 ./ expected)
p = 1 - chi2cdf(chi2stat,1)


%% Choice curves grouped by biased item and by how effective the
% manipulation was.

aggregCountDecisionsLeftBiased = cell(3,1);
aggregCountLeftChosenLeftBiased = cell(3,1);
aggregCountDecisionsRightBiased = cell(3,1);
aggregCountLeftChosenRightBiased = cell(3,1);
for i=1:3
    aggregCountDecisionsLeftBiased{i} = zeros(length(valueDiffs), 1);
    aggregCountLeftChosenLeftBiased{i} = zeros(length(valueDiffs), 1);
    aggregCountDecisionsRightBiased{i} = zeros(length(valueDiffs), 1);
    aggregCountLeftChosenRightBiased{i} = zeros(length(valueDiffs), 1);
end

for i=1:length(parcodes)
    for s=sessions
        [countDecisionsLeftBiased, countLeftChosenLeftBiased, ...
            countDecisionsRightBiased, countLeftChosenRightBiased] = ...
            ChoicesGroupedByBiasedAndRank(valueDiffs, parcodes(i,:), s);

        for j=1:3
            for k=1:length(valueDiffs)
                aggregCountDecisionsLeftBiased{j}(k) = ...
                    aggregCountDecisionsLeftBiased{j}(k) + ...
                    countDecisionsLeftBiased{j}(k);
                aggregCountLeftChosenLeftBiased{j}(k) = ...
                    aggregCountLeftChosenLeftBiased{j}(k) + ...
                    countLeftChosenLeftBiased{j}(k);
                aggregCountDecisionsRightBiased{j}(k) = ...
                    aggregCountDecisionsRightBiased{j}(k) + ...
                    countDecisionsRightBiased{j}(k);
                aggregCountLeftChosenRightBiased{j}(k) = ...
                    aggregCountLeftChosenRightBiased{j}(k) + ...
                    countLeftChosenRightBiased{j}(k);
            end
        end
    end
end

% Plot aggregated choice curves.
figure; hold;
c = [0.7 * cyan; 0.1 + 0.9 * cyan; 0.6 + 0.4 * cyan];
for i=1:3
    aggregProbLeftChosenLeftBiased = ...
        aggregCountLeftChosenLeftBiased{i} ./ ...
        aggregCountDecisionsLeftBiased{i};
    stdProbLeftChosenLeftBiased = sqrt( ...
        aggregProbLeftChosenLeftBiased .* ...
        (1-aggregProbLeftChosenLeftBiased) ./ ...
        aggregCountDecisionsLeftBiased{i});
    errorbar(plotValueDiffs, aggregProbLeftChosenLeftBiased, ...
        stdProbLeftChosenLeftBiased, 'LineStyle', '-', 'Marker', '.', ...
        'MarkerSize', 25, 'Color', c(i,:), 'LineWidth', 2);
end

c = [0.7 * yellow; 0.1 + 0.9 * yellow; 0.6 + 0.4 * yellow];
for i=1:3
    aggregProbLeftChosenRightBiased = ...
        aggregCountLeftChosenRightBiased{i} ./ ...
        aggregCountDecisionsRightBiased{i};
    stdProbLeftChosenRightBiased = sqrt( ...
        aggregProbLeftChosenRightBiased .* ...
        (1-aggregProbLeftChosenRightBiased) ./ ...
        aggregCountDecisionsRightBiased{i});
    errorbar(plotValueDiffs, aggregProbLeftChosenRightBiased, ...
        stdProbLeftChosenRightBiased, 'LineStyle', '-', 'Marker', '.', ...
        'MarkerSize', 25, 'Color', c(i,:), 'LineWidth', 2);
end

line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5]);
xlim(xLim);
ylim([0 1]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
legend('Left biased 1st', 'Left biased 2nd', 'Left biased 3rd', ...
    'Right biased 1st', 'Right biased 2nd', 'Right biased 3rd', ...
    'Location', 'Southeast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Choice curves grouped by longest fixated item.

aggregCountDecisionsLeftBiased = zeros(length(valueDiffs),1);
aggregCountLeftChosenLeftBiased = zeros(length(valueDiffs),1);
aggregCountDecisionsRightBiased = zeros(length(valueDiffs),1);
aggregCountLeftChosenRightBiased = zeros(length(valueDiffs),1);

for i=1:length(parcodes)
    subjCountDecisionsLeftBiased = zeros(length(valueDiffs),1);
    subjCountLeftChosenLeftBiased = zeros(length(valueDiffs),1);
    subjCountDecisionsRightBiased = zeros(length(valueDiffs),1);
    subjCountLeftChosenRightBiased = zeros(length(valueDiffs),1);
    
    for s=sessions
        [countDecisionsLeftBiased, countLeftChosenLeftBiased, ...
            countDecisionsRightBiased, countLeftChosenRightBiased] = ...
            ChoicesGroupedByLongestFixated(valueDiffs, parcodes(i,:), s);
        subjCountDecisionsLeftBiased = subjCountDecisionsLeftBiased + ...
            countDecisionsLeftBiased;
        subjCountLeftChosenLeftBiased = subjCountLeftChosenLeftBiased + ...
            countLeftChosenLeftBiased;
        subjCountDecisionsRightBiased = subjCountDecisionsRightBiased + ...
            countDecisionsRightBiased;
        subjCountLeftChosenRightBiased = ...
            subjCountLeftChosenRightBiased + countLeftChosenRightBiased;
    end
    
    subjProbLeftChosenLeftBiased = subjCountLeftChosenLeftBiased ./ ...
        subjCountDecisionsLeftBiased;
    subjProbLeftChosenRightBiased = subjCountLeftChosenRightBiased ./ ...
        subjCountDecisionsRightBiased;

    aggregCountDecisionsLeftBiased = aggregCountDecisionsLeftBiased + ...
        subjCountDecisionsLeftBiased;
    aggregCountLeftChosenLeftBiased = aggregCountLeftChosenLeftBiased + ...
        subjCountLeftChosenLeftBiased;
    aggregCountDecisionsRightBiased = aggregCountDecisionsRightBiased + ...
        subjCountDecisionsRightBiased;
    aggregCountLeftChosenRightBiased = ...
        aggregCountLeftChosenRightBiased + subjCountLeftChosenRightBiased;
end

% Plot aggregated choice curves.
figure; hold;
aggregProbLeftChosenLeftBiased = aggregCountLeftChosenLeftBiased ./ ...
    aggregCountDecisionsLeftBiased;
stdProbLeftChosenLeftBiased = sqrt(aggregProbLeftChosenLeftBiased .* ...
    (1-aggregProbLeftChosenLeftBiased));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenLeftBiased, ...
    aggregCountDecisionsLeftBiased);
shadedErrorBar(plotValueDiffs, aggregProbLeftChosenLeftBiased, confInt, ...
    {'-', 'Marker', '.', 'MarkerSize', 24, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

aggregProbLeftChosenRightBiased = aggregCountLeftChosenRightBiased ./ ...
    aggregCountDecisionsRightBiased;
stdProbLeftChosenRightBiased = sqrt(aggregProbLeftChosenRightBiased .* ...
    (1-aggregProbLeftChosenRightBiased));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenRightBiased, ...
    aggregCountDecisionsRightBiased);
shadedErrorBar(plotValueDiffs, aggregProbLeftChosenRightBiased, ...
    confInt, {'-', 'Marker', '.', 'MarkerSize', 24, 'Color', yellow, ...
    'LineWidth', 3}, transparency);

line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5]);
xlim(xLim);
ylim([0 1]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
legend('Left most fix', 'Right most fix', 'Location', 'Southeast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Choice curves grouped by first fixated item.

aggregCountDecisionsLeftFirstFix = zeros(length(valueDiffs),1);
aggregCountLeftChosenLeftFirstFix = zeros(length(valueDiffs),1);
aggregCountDecisionsRightFirstFix = zeros(length(valueDiffs),1);
aggregCountLeftChosenRightFirstFix = zeros(length(valueDiffs),1);

for i=1:length(parcodes)
    subjCountDecisionsLeftFirstFix = zeros(length(valueDiffs),1);
    subjCountLeftChosenLeftFirstFix = zeros(length(valueDiffs),1);
    subjCountDecisionsRightFirstFix = zeros(length(valueDiffs),1);
    subjCountLeftChosenRightFirstFix = zeros(length(valueDiffs),1);
    
    for s=sessions
        [countDecisionsLeftFirstFix, countLeftChosenLeftFirstFix, ...
            countDecisionsRightFirstFix, ...
            countLeftChosenRightFirstFix] = ...
            ChoicesGroupedByFirstFixation(valueDiffs, parcodes(i,:), s);
        subjCountDecisionsLeftFirstFix = ...
            subjCountDecisionsLeftFirstFix + countDecisionsLeftFirstFix;
        subjCountLeftChosenLeftFirstFix = ...
            subjCountLeftChosenLeftFirstFix + countLeftChosenLeftFirstFix;
        subjCountDecisionsRightFirstFix = ...
            subjCountDecisionsRightFirstFix + countDecisionsRightFirstFix;
        subjCountLeftChosenRightFirstFix = ...
            subjCountLeftChosenRightFirstFix + ...
            countLeftChosenRightFirstFix;
    end
    
    subjProbLeftChosenLeftFirstFix = subjCountLeftChosenLeftFirstFix ./ ...
        subjCountDecisionsLeftFirstFix;
    subjProbLeftChosenRightFirstFix = ...
        subjCountLeftChosenRightFirstFix ./ ...
        subjCountDecisionsRightFirstFix;

    aggregCountDecisionsLeftFirstFix = ...
        aggregCountDecisionsLeftFirstFix + subjCountDecisionsLeftFirstFix;
    aggregCountLeftChosenLeftFirstFix = ...
        aggregCountLeftChosenLeftFirstFix + ...
        subjCountLeftChosenLeftFirstFix;
    aggregCountDecisionsRightFirstFix = ...
        aggregCountDecisionsRightFirstFix + ...
        subjCountDecisionsRightFirstFix;
    aggregCountLeftChosenRightFirstFix = ...
        aggregCountLeftChosenRightFirstFix + ...
        subjCountLeftChosenRightFirstFix;
end

% Plot aggregated choice curves.
figure; hold;
aggregProbLeftChosenLeftFirstFix = aggregCountLeftChosenLeftFirstFix ./ ...
    aggregCountDecisionsLeftFirstFix;
stdProbLeftChosenLeftFirstFix = sqrt( ...
    aggregProbLeftChosenLeftFirstFix .* ...
    (1-aggregProbLeftChosenLeftFirstFix));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenLeftFirstFix, ...
    aggregCountDecisionsLeftFirstFix);
shadedErrorBar(plotValueDiffs, aggregProbLeftChosenLeftFirstFix, ...
    confInt, {'-', 'Marker', '.', 'MarkerSize', 24, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

aggregProbLeftChosenRightFirstFix = ...
    aggregCountLeftChosenRightFirstFix ./ ...
    aggregCountDecisionsRightFirstFix;
stdProbLeftChosenRightFirstFix = sqrt( ...
    aggregProbLeftChosenRightFirstFix .* ...
    (1-aggregProbLeftChosenRightFirstFix));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenRightFirstFix, ...
    aggregCountDecisionsRightFirstFix);
shadedErrorBar(plotValueDiffs, aggregProbLeftChosenRightFirstFix, ...
    confInt, {'-', 'Marker', '.', 'MarkerSize', 24, 'Color', yellow, ...
    'LineWidth', 3}, transparency);

line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5]);
xlim(xLim);
ylim([0 1]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
legend('Left first fix', 'Right first fix', 'Location', 'Southeast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Choice curves grouped by biased item only when longest fixated.

aggregCountDecisionsLeftBiased = zeros(length(valueDiffs),1);
aggregCountLeftChosenLeftBiased = zeros(length(valueDiffs),1);
aggregCountDecisionsRightBiased = zeros(length(valueDiffs),1);
aggregCountLeftChosenRightBiased = zeros(length(valueDiffs),1);

for i=1:length(parcodes)
    subjCountDecisionsLeftBiased = zeros(length(valueDiffs),1);
    subjCountLeftChosenLeftBiased = zeros(length(valueDiffs),1);
    subjCountDecisionsRightBiased = zeros(length(valueDiffs),1);
    subjCountLeftChosenRightBiased = zeros(length(valueDiffs),1);
    
    for s=sessions
        [countDecisionsLeftBiased, countLeftChosenLeftBiased, ...
            countDecisionsRightBiased, countLeftChosenRightBiased] = ...
            ChoicesGroupedByBiasedLongestFixated(valueDiffs, ...
            parcodes(i,:), s);
        subjCountDecisionsLeftBiased = subjCountDecisionsLeftBiased + ...
            countDecisionsLeftBiased;
        subjCountLeftChosenLeftBiased = subjCountLeftChosenLeftBiased + ...
            countLeftChosenLeftBiased;
        subjCountDecisionsRightBiased = subjCountDecisionsRightBiased + ...
            countDecisionsRightBiased;
        subjCountLeftChosenRightBiased = ...
            subjCountLeftChosenRightBiased + countLeftChosenRightBiased;
    end
    
    subjProbLeftChosenLeftBiased = subjCountLeftChosenLeftBiased ./ ...
        subjCountDecisionsLeftBiased;
    subjProbLeftChosenRightBiased = subjCountLeftChosenRightBiased ./ ...
        subjCountDecisionsRightBiased;

    aggregCountDecisionsLeftBiased = aggregCountDecisionsLeftBiased + ...
        subjCountDecisionsLeftBiased;
    aggregCountLeftChosenLeftBiased = aggregCountLeftChosenLeftBiased + ...
        subjCountLeftChosenLeftBiased;
    aggregCountDecisionsRightBiased = aggregCountDecisionsRightBiased + ...
        subjCountDecisionsRightBiased;
    aggregCountLeftChosenRightBiased = ...
        aggregCountLeftChosenRightBiased + subjCountLeftChosenRightBiased;
end

% Plot aggregated choice curves.
figure; hold;
aggregProbLeftChosenLeftBiased = aggregCountLeftChosenLeftBiased ./ ...
    aggregCountDecisionsLeftBiased;
stdProbLeftChosenLeftBiased = sqrt(aggregProbLeftChosenLeftBiased .* ...
    (1-aggregProbLeftChosenLeftBiased));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenLeftBiased, ...
    aggregCountDecisionsLeftBiased);
shadedErrorBar(valueDiffs, aggregProbLeftChosenLeftBiased, ...
    confInt, {'-', 'Marker', '.', 'MarkerSize', 24, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

aggregProbLeftChosenRightBiased = aggregCountLeftChosenRightBiased ./ ...
    aggregCountDecisionsRightBiased;
stdProbLeftChosenRightBiased = sqrt(aggregProbLeftChosenRightBiased .* ...
    (1-aggregProbLeftChosenRightBiased));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenRightBiased, ...
    aggregCountDecisionsRightBiased);
shadedErrorBar(valueDiffs, aggregProbLeftChosenRightBiased, ...
    confInt, {'-', 'Marker', '.', 'MarkerSize', 24, 'Color', yellow, ...
    'LineWidth', 3}, transparency);

line([-16 16],[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5], ...
    'LineWidth', 2);
xlim([-16 16]);
ylim([0 1]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
legend('L bias-target, longest fix', 'R bias-target, longest fix', ...
    'Location', 'Southeast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Choice curves grouped by biased item only when least fixated.

aggregCountDecisionsLeftBiased = zeros(length(valueDiffs),1);
aggregCountLeftChosenLeftBiased = zeros(length(valueDiffs),1);
aggregCountDecisionsRightBiased = zeros(length(valueDiffs),1);
aggregCountLeftChosenRightBiased = zeros(length(valueDiffs),1);

for i=1:length(parcodes)
    subjCountDecisionsLeftBiased = zeros(length(valueDiffs),1);
    subjCountLeftChosenLeftBiased = zeros(length(valueDiffs),1);
    subjCountDecisionsRightBiased = zeros(length(valueDiffs),1);
    subjCountLeftChosenRightBiased = zeros(length(valueDiffs),1);
    
    for s=sessions
        [countDecisionsLeftBiased, countLeftChosenLeftBiased, ...
            countDecisionsRightBiased, countLeftChosenRightBiased] = ...
            ChoicesGroupedByBiasedLeastFixated(valueDiffs, ...
            parcodes(i,:), s);
        subjCountDecisionsLeftBiased = subjCountDecisionsLeftBiased + ...
            countDecisionsLeftBiased;
        subjCountLeftChosenLeftBiased = subjCountLeftChosenLeftBiased + ...
            countLeftChosenLeftBiased;
        subjCountDecisionsRightBiased = subjCountDecisionsRightBiased + ...
            countDecisionsRightBiased;
        subjCountLeftChosenRightBiased = ...
            subjCountLeftChosenRightBiased + countLeftChosenRightBiased;
    end
    
    subjProbLeftChosenLeftBiased = subjCountLeftChosenLeftBiased ./ ...
        subjCountDecisionsLeftBiased;
    subjProbLeftChosenRightBiased = subjCountLeftChosenRightBiased ./ ...
        subjCountDecisionsRightBiased;

    aggregCountDecisionsLeftBiased = aggregCountDecisionsLeftBiased + ...
        subjCountDecisionsLeftBiased;
    aggregCountLeftChosenLeftBiased = aggregCountLeftChosenLeftBiased + ...
        subjCountLeftChosenLeftBiased;
    aggregCountDecisionsRightBiased = aggregCountDecisionsRightBiased + ...
        subjCountDecisionsRightBiased;
    aggregCountLeftChosenRightBiased = ...
        aggregCountLeftChosenRightBiased + subjCountLeftChosenRightBiased;
end

% Plot aggregated choice curves.
figure; hold;
aggregProbLeftChosenLeftBiased = aggregCountLeftChosenLeftBiased ./ ...
    aggregCountDecisionsLeftBiased;
stdProbLeftChosenLeftBiased = sqrt(aggregProbLeftChosenLeftBiased .* ...
    (1-aggregProbLeftChosenLeftBiased));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenLeftBiased, ...
    aggregCountDecisionsLeftBiased);
shadedErrorBar(valueDiffs, aggregProbLeftChosenLeftBiased, ...
    confInt, {'-', 'Marker', '.', 'MarkerSize', 24, 'Color', cyan, ...
    'LineWidth', 3}, transparency);

aggregProbLeftChosenRightBiased = aggregCountLeftChosenRightBiased ./ ...
    aggregCountDecisionsRightBiased;
stdProbLeftChosenRightBiased = sqrt(aggregProbLeftChosenRightBiased .* ...
    (1-aggregProbLeftChosenRightBiased));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenRightBiased, ...
    aggregCountDecisionsRightBiased);
shadedErrorBar(valueDiffs, aggregProbLeftChosenRightBiased, ...
    confInt, {'-', 'Marker', '.', 'MarkerSize', 24, 'Color', yellow, ...
    'LineWidth', 3}, transparency);

line([-16 16],[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5], ...
    'LineWidth', 2);
xlim([-16 16]);
ylim([0 1]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
legend('L bias-target, least fix', 'R bias-target, least fix', ...
    'Location', 'Southeast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Compare choice curves for bias longest fixated and biased least fixated.

aggregCountDecisionsLeftBiased = zeros(length(valueDiffs),1);
aggregCountLeftChosenLeftBiased = zeros(length(valueDiffs),1);
aggregCountDecisionsRightBiased = zeros(length(valueDiffs),1);
aggregCountLeftChosenRightBiased = zeros(length(valueDiffs),1);

for i=1:length(parcodes)
    subjCountDecisionsLeftBiased = zeros(length(valueDiffs),1);
    subjCountLeftChosenLeftBiased = zeros(length(valueDiffs),1);
    subjCountDecisionsRightBiased = zeros(length(valueDiffs),1);
    subjCountLeftChosenRightBiased = zeros(length(valueDiffs),1);
    
    for s=sessions
        [countDecisionsLeftBiased, countLeftChosenLeftBiased, ...
            countDecisionsRightBiased, countLeftChosenRightBiased] = ...
            ChoicesGroupedByBiasedLongestFixated(valueDiffs, ...
            parcodes(i,:), s);
        subjCountDecisionsLeftBiased = subjCountDecisionsLeftBiased + ...
            countDecisionsLeftBiased;
        subjCountLeftChosenLeftBiased = subjCountLeftChosenLeftBiased + ...
            countLeftChosenLeftBiased;
        subjCountDecisionsRightBiased = subjCountDecisionsRightBiased + ...
            countDecisionsRightBiased;
        subjCountLeftChosenRightBiased = ...
            subjCountLeftChosenRightBiased + countLeftChosenRightBiased;
    end
    
    subjProbLeftChosenLeftBiased = subjCountLeftChosenLeftBiased ./ ...
        subjCountDecisionsLeftBiased;
    subjProbLeftChosenRightBiased = subjCountLeftChosenRightBiased ./ ...
        subjCountDecisionsRightBiased;

    aggregCountDecisionsLeftBiased = aggregCountDecisionsLeftBiased + ...
        subjCountDecisionsLeftBiased;
    aggregCountLeftChosenLeftBiased = aggregCountLeftChosenLeftBiased + ...
        subjCountLeftChosenLeftBiased;
    aggregCountDecisionsRightBiased = aggregCountDecisionsRightBiased + ...
        subjCountDecisionsRightBiased;
    aggregCountLeftChosenRightBiased = ...
        aggregCountLeftChosenRightBiased + subjCountLeftChosenRightBiased;
end

% Plot aggregated choice curves for biased longest fixated.
figure; hold;
aggregProbLeftChosenLeftBiased = aggregCountLeftChosenLeftBiased ./ ...
    aggregCountDecisionsLeftBiased;
stdProbLeftChosenLeftBiased = sqrt(aggregProbLeftChosenLeftBiased .* ...
    (1-aggregProbLeftChosenLeftBiased));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenLeftBiased, ...
    aggregCountDecisionsLeftBiased);
errorbar(valueDiffs, aggregProbLeftChosenLeftBiased, ...
    confInt, '-', 'Marker', 'o', 'MarkerSize', 8, 'Color', cyan, ...
    'LineWidth', 3, 'MarkerFaceColor', cyan);

aggregProbLeftChosenRightBiased = aggregCountLeftChosenRightBiased ./ ...
    aggregCountDecisionsRightBiased;
stdProbLeftChosenRightBiased = sqrt(aggregProbLeftChosenRightBiased .* ...
    (1-aggregProbLeftChosenRightBiased));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenRightBiased, ...
    aggregCountDecisionsRightBiased);
errorbar(valueDiffs, aggregProbLeftChosenRightBiased, ...
    confInt, '-', 'Marker', 'o', 'MarkerSize', 8, 'Color', yellow, ...
    'LineWidth', 3, 'MarkerFaceColor', yellow);

aggregCountDecisionsLeftBiased = zeros(length(valueDiffs),1);
aggregCountLeftChosenLeftBiased = zeros(length(valueDiffs),1);
aggregCountDecisionsRightBiased = zeros(length(valueDiffs),1);
aggregCountLeftChosenRightBiased = zeros(length(valueDiffs),1);

for i=1:length(parcodes)
    subjCountDecisionsLeftBiased = zeros(length(valueDiffs),1);
    subjCountLeftChosenLeftBiased = zeros(length(valueDiffs),1);
    subjCountDecisionsRightBiased = zeros(length(valueDiffs),1);
    subjCountLeftChosenRightBiased = zeros(length(valueDiffs),1);
    
    for s=sessions
        [countDecisionsLeftBiased, countLeftChosenLeftBiased, ...
            countDecisionsRightBiased, countLeftChosenRightBiased] = ...
            ChoicesGroupedByBiasedLeastFixated(valueDiffs, ...
            parcodes(i,:), s);
        subjCountDecisionsLeftBiased = subjCountDecisionsLeftBiased + ...
            countDecisionsLeftBiased;
        subjCountLeftChosenLeftBiased = subjCountLeftChosenLeftBiased + ...
            countLeftChosenLeftBiased;
        subjCountDecisionsRightBiased = subjCountDecisionsRightBiased + ...
            countDecisionsRightBiased;
        subjCountLeftChosenRightBiased = ...
            subjCountLeftChosenRightBiased + countLeftChosenRightBiased;
    end
    
    subjProbLeftChosenLeftBiased = subjCountLeftChosenLeftBiased ./ ...
        subjCountDecisionsLeftBiased;
    subjProbLeftChosenRightBiased = subjCountLeftChosenRightBiased ./ ...
        subjCountDecisionsRightBiased;

    aggregCountDecisionsLeftBiased = aggregCountDecisionsLeftBiased + ...
        subjCountDecisionsLeftBiased;
    aggregCountLeftChosenLeftBiased = aggregCountLeftChosenLeftBiased + ...
        subjCountLeftChosenLeftBiased;
    aggregCountDecisionsRightBiased = aggregCountDecisionsRightBiased + ...
        subjCountDecisionsRightBiased;
    aggregCountLeftChosenRightBiased = ...
        aggregCountLeftChosenRightBiased + subjCountLeftChosenRightBiased;
end

% Plot aggregated choice curves for biased least fixated.
aggregProbLeftChosenLeftBiased = aggregCountLeftChosenLeftBiased ./ ...
    aggregCountDecisionsLeftBiased;
stdProbLeftChosenLeftBiased = sqrt(aggregProbLeftChosenLeftBiased .* ...
    (1-aggregProbLeftChosenLeftBiased));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenLeftBiased, ...
    aggregCountDecisionsLeftBiased);
errorbar(valueDiffs, aggregProbLeftChosenLeftBiased, ...
    confInt, '--', 'Marker', 'd', 'MarkerSize', 8, 'Color', cyan, ...
    'LineWidth', 3, 'MarkerFaceColor', cyan);

aggregProbLeftChosenRightBiased = aggregCountLeftChosenRightBiased ./ ...
    aggregCountDecisionsRightBiased;
stdProbLeftChosenRightBiased = sqrt(aggregProbLeftChosenRightBiased .* ...
    (1-aggregProbLeftChosenRightBiased));
confInt = Get95PercConfidenceInterval(stdProbLeftChosenRightBiased, ...
    aggregCountDecisionsRightBiased);
errorbar(valueDiffs, aggregProbLeftChosenRightBiased, ...
    confInt, '--', 'Marker', 'd', 'MarkerSize', 8, 'Color', yellow, ...
    'LineWidth', 3, 'MarkerFaceColor', yellow);

line([-16 16],[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5], ...
    'LineWidth', 2);
xlim([-16 16]);
ylim([0 1]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('P(left chosen)','FontSize', 24);
legend('Left biased, longest fix', 'Right biased, longest fix', ...
    'Left biased, least fix', 'Right biased, least fix', 'Location', ...
    'SouthEast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Obtain trial duration as function of value difference.

aggregTrialDuration = cell(length(valueDiffs), 1);
figure; hold all;

for i=1:length(parcodes)
    subjTrialDuration = cell(length(valueDiffs), 1);

    for s=sessions
        trialDuration = TrialDurationFromEyetrackerByValueDiff( ...
            valueDiffs, [], [], [], parcodes(i,:), s);

        for j=1:length(valueDiffs)
            subjTrialDuration{j} = [subjTrialDuration{j}; ...
                trialDuration{j}];
        end
    end
    
    for j=1:length(valueDiffs)
        aggregTrialDuration{j} = [aggregTrialDuration{j}; ...
            subjTrialDuration{j}];
    end

    % Plot individual trial durations.
    meanRt = zeros(length(valueDiffs),1);
    stdRt = zeros(length(valueDiffs),1);
    for j=1:length(valueDiffs)
        meanRt(j) = mean(subjTrialDuration{j});
        stdRt(j) = std(subjTrialDuration{j}) / ...
            sqrt(length(subjTrialDuration{j}));
    end
    c = alpha + (1-alpha) * colors(i,:);
    plot(plotValueDiffs, meanRt, '--.', 'MarkerSize', 25, 'Color', c);
end

% Plot aggregated trial durations.
meanAggregTrialDuration = zeros(length(valueDiffs),1);
stdAggregTrialDuration = zeros(length(valueDiffs),1);
for j=1:length(valueDiffs)
    meanAggregTrialDuration(j) = mean(aggregTrialDuration{j});
    stdAggregTrialDuration(j) = std(aggregTrialDuration{j}) / ...
        sqrt(length(aggregTrialDuration{j}));
end
errorbar(plotValueDiffs, meanAggregTrialDuration, ...
    stdAggregTrialDuration, 'LineStyle', '-', 'Marker', '.', ...
    'MarkerSize', 25, 'Color', 'k', 'LineWidth', 2);
xlim(xLim);
ylim([0 4000]);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('Trial duration [ms]','FontSize', 24);
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Obtain reaction time as function of best value minus worst value.

aggregReactiontime = cell(length(nonNegativeValueDiffs), 1);
figure; hold on;

for i=1:length(parcodes)
    subjReactionTime = cell(length(nonNegativeValueDiffs), 1);

    for s=sessions
        reactionTime = ReactionTimesByValueBestMinusWorst( ...
            nonNegativeValueDiffs, parcodes(i,:), s);

        for j=1:length(nonNegativeValueDiffs)
            subjReactionTime{j} = [subjReactionTime{j}; reactionTime{j}];
        end
    end
    
    for j=1:length(nonNegativeValueDiffs)
        aggregReactiontime{j} = [aggregReactiontime{j}; ...
            subjReactionTime{j}];
    end

    % Plot individual reaction times.
    meanRt = zeros(length(nonNegativeValueDiffs),1);
    stdRt = zeros(length(nonNegativeValueDiffs),1);
    for j=1:length(nonNegativeValueDiffs)
        meanRt(j) = mean(subjReactionTime{j});
        stdRt(j) = std(subjReactionTime{j}) / ...
            sqrt(length(subjReactionTime{j}));
    end
    c = alpha + (1-alpha) * colors(i,:);
    plot(plotNonNegativeValueDiffs, meanRt, '--.', 'MarkerSize', 25, ...
        'Color', c);
end

% Plot aggregated reaction times.
meanAggregReactionTime = zeros(length(nonNegativeValueDiffs),1);
stdAggregReactionTime = zeros(length(nonNegativeValueDiffs),1);
for j=1:length(nonNegativeValueDiffs)
    meanAggregReactionTime(j) = mean(aggregReactiontime{j});
    stdAggregReactionTime(j) = std(aggregReactiontime{j}) / ...
        sqrt(length(aggregReactiontime{j}));
end
errorbar(plotNonNegativeValueDiffs, meanAggregReactionTime, ...
    stdAggregReactionTime, 'LineStyle', '-', 'Marker', '.', ...
    'MarkerSize', 25, 'Color', 'k', 'LineWidth', 2);
xlim([-0.5 3.5]);
ylim([0 1600]);
xlabel('r_{closest} - r_{farthest}','FontSize', 24);
ylabel('Reaction time [ms]','FontSize', 24);
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Obtain time advantage of left item as function of value difference,
% for original task and for causality task grouped by biased item.

% Get time advantage for original experiment.
aggregTimeAdvantageLeft = cell(length(valueDiffs), 1);

for i=1:length(oldParcodes)
    for s=oldSessions
        timeAdvantageLeft = TimeAdvantageLeftVsValueDifference( ...
            valueDiffs, oldParcodes(i,:), s);
        for j=1:length(valueDiffs)
            aggregTimeAdvantageLeft{j} = [aggregTimeAdvantageLeft{j}; ...
                timeAdvantageLeft{j}];
        end
    end
end

% Get time advantage for causality experiment.
aggregTimeAdvantageLeftBiasedLeft = cell(length(valueDiffs), 1);
aggregTimeAdvantageLeftBiasedRight = cell(length(valueDiffs), 1);

for i=1:length(parcodes)
    for s=sessions
        [timeAdvantageLeftBiasedLeft, timeAdvantageLeftBiasedRight] = ...
            TimeAdvantageLeftVsValueDiffGroupedByBiased( ...
            valueDiffs, parcodes(i,:), s);
        for j=1:length(valueDiffs)
            aggregTimeAdvantageLeftBiasedLeft{j} = ...
                [aggregTimeAdvantageLeftBiasedLeft{j}; ...
                timeAdvantageLeftBiasedLeft{j}];
            aggregTimeAdvantageLeftBiasedRight{j} = ...
                [aggregTimeAdvantageLeftBiasedRight{j}; ...
                timeAdvantageLeftBiasedRight{j}];
        end
    end
end

% Generate plots.
meanTimeAdvantageLeft = zeros(length(valueDiffs),1);
confIntTimeAdvantageLeft = zeros(length(valueDiffs),1);
meanTimeAdvantageLeftBiasedLeft = zeros(length(valueDiffs),1);
confIntTimeAdvantageLeftBiasedLeft = zeros(length(valueDiffs),1);
meanTimeAdvantageLeftBiasedRight = zeros(length(valueDiffs),1);
confIntTimeAdvantageLeftBiasedRight = zeros(length(valueDiffs),1);
for j=1:length(valueDiffs)
    meanTimeAdvantageLeft(j) = mean(aggregTimeAdvantageLeft{j});
    confIntTimeAdvantageLeft(j) = Get95PercConfidenceInterval( ...
        std(aggregTimeAdvantageLeft{j}), ...
        length(aggregTimeAdvantageLeft{j}));
    meanTimeAdvantageLeftBiasedLeft(j) = ...
        mean(aggregTimeAdvantageLeftBiasedLeft{j});
    confIntTimeAdvantageLeftBiasedLeft(j) = ...
        Get95PercConfidenceInterval( ...
        std(aggregTimeAdvantageLeftBiasedLeft{j}), ...
        length(aggregTimeAdvantageLeftBiasedLeft{j}));
    meanTimeAdvantageLeftBiasedRight(j) = ...
        mean(aggregTimeAdvantageLeftBiasedRight{j});
    confIntTimeAdvantageLeftBiasedRight(j) = ...
        Get95PercConfidenceInterval( ...
        std(aggregTimeAdvantageLeftBiasedRight{j}), ...
        length(aggregTimeAdvantageLeftBiasedRight{j}));
end

figure; hold;
shadedErrorBar(plotValueDiffs, meanTimeAdvantageLeft, ...
    confIntTimeAdvantageLeft, {'-', 'Marker', '.', 'MarkerSize', 24, ...
    'Color', green, 'LineWidth', 3}, transparency);
shadedErrorBar(plotValueDiffs, meanTimeAdvantageLeftBiasedLeft, ...
    confIntTimeAdvantageLeftBiasedLeft, {'-', 'Marker', '.', ...
    'MarkerSize', 24, 'Color', cyan, 'LineWidth', 3}, transparency);
shadedErrorBar(plotValueDiffs, meanTimeAdvantageLeftBiasedRight, ...
    confIntTimeAdvantageLeftBiasedRight, {'-', 'Marker', '.', ...
    'MarkerSize', 24, 'Color', yellow, 'LineWidth', 3}, transparency);
line(xLim,[0.5 0.5],'LineStyle','--','Color',[0.5 0.5 0.5], ...
    'LineWidth', 2);
xlim(xLim);
xlabel('r_{left} - r_{right}','FontSize', 24);
ylabel('Overall relative fixation left [ms]','FontSize', 24);
leg = legend('Exp 1', 'Exp 2 L bias-target', 'Exp 2 R bias-target', ...
    'Location', 'SouthEast');
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Mean total fixation duration for each item on effective trials.

% Get time advantage for causality experiment.
aggregTimeFixBiased = [];
aggregTimeFixUnbiased = [];

for i=1:length(parcodes)
    for s=sessions
        [timeFixBiased, timeFixUnbiased] = ...
            TimeFixBiasedAndUnbiasedItems(parcodes(i,:), s);
        aggregTimeFixBiased = [aggregTimeFixBiased; timeFixBiased];
        aggregTimeFixUnbiased = [aggregTimeFixUnbiased; timeFixUnbiased];
    end
end

means = zeros(1,2);
means(1,1) = mean(aggregTimeFixBiased);
means(1,2) = mean(aggregTimeFixUnbiased);
% confInts = zeros(1,2);
% confInts(1,1) = Get95PercConfidenceInterval(std(aggregTimeFixBiased), ...
%     length(aggregTimeFixBiased));
% confInts(1,2) = Get95PercConfidenceInterval(std(aggregTimeFixUnbiased), ...
%     length(aggregTimeFixUnbiased));

stds = zeros(1,2);
stds(1,1) = std(aggregTimeFixBiased);
stds(1,2) = std(aggregTimeFixUnbiased);

figure; hold on;
bar(means, 'FaceColor', 0.7 + 0.3 * cyan, 'BarWidth', 0.4, ...
    'EdgeColor', 0.7 * cyan, 'BarWidth', 0.4, 'LineWidth', 3);
errorbar(means, stds, '.', 'Color', 0.7 * cyan, 'LineWidth', 3);
set(gca, 'XTick', 1:2, 'XTickLabel', {'Biased item', 'Unbiased item'});
ylabel('Mean fixation time [ms]', 'FontSize', 24);
title('Total fixation time on effective trials', 'FontSize', 24);
set(gca, 'FontSize', 20);
xlim([0.5 2.5]);
orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Corrected probability of choosing left item vs. time advantage of left
% item, comparing Exp 1 and Exp 2.

timeBins = -600:200:600;

% Get probabilities for Exp 1.
avgProbs = GetAvgProbChooseLeftByValueDiff(valueDiffs, [], [], [], ...
    oldParcodes, oldSessions);
corrProbs = CorrProbChooseLeftVsTimeAdvantageLeft(timeBins, valueDiffs, ...
    avgProbs, [], [], [], [], [], oldParcodes, oldSessions);
means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(corrProbs)
    means(j) = mean(corrProbs{j});
    stdev = std(corrProbs{j});
    confInts(j) = Get95PercConfidenceInterval(stdev, length(corrProbs{j}));
end

figure; hold;
h1 = errorbar(1:length(timeBins), means, confInts, '-', 'Marker', '.', ...
    'MarkerSize', 25, 'Color', green, 'LineWidth', 3);

% Get probabilities for Exp 2.
avgProbs = GetAvgProbChooseLeftByValueDiff(valueDiffs, [], [], [], ...
    parcodes, sessions, false, false, true);
corrProbs = CorrProbChooseLeftVsTimeAdvantageLeft(timeBins, valueDiffs, ...
    avgProbs, [], [], [], [], [], parcodes, sessions, false, false, true);
means = zeros(length(timeBins),1);
confInts = zeros(length(timeBins),1);
for j=1:length(corrProbs)
    means(j) = mean(corrProbs{j});
    stdev = std(corrProbs{j});
    confInts(j) = Get95PercConfidenceInterval(stdev, length(corrProbs{j}));
end

h2 = errorbar(1:length(timeBins), means, confInts, '-', 'Marker', '.', ...
    'MarkerSize', 25, 'Color', pink, 'LineWidth', 3);

xlabel('Overall relative fixation left [ms]', 'FontSize', 24);
ylabel('Corrected p(choose left)','FontSize', 24);
legend([h1 h2], {'Exp 1', 'Exp 2'}, 'Location', 'SouthEast');
xlim([0 length(timeBins)+1]);
line([0 length(timeBins)+1],[0 0],'LineStyle','--','Color', ...
    [0.5 0.5 0.5],'LineWidth',2);

xlabels = cell(length(timeBins), 1);
for i=1:length(timeBins)
    xlabels{i,1} = sprintf('%d', timeBins(i));
end
set(gca, 'XTick', 1:length(timeBins), 'XTickLabel', xlabels);
set(gca,'FontSize',20);

orient landscape
print('-dpsc2', '-append', plotsFileName);


%% Get mean and median of fixation times on bias-target and non-bias-target
% for trials where the bias-target was longest and least fixated.

[fixTimesBiased, fixTimeUnbiased] = ...
    FixTimesWhenBiasedLongestFixated(parcodes, sessions);
medianBiasedWhenBiasedLongestFix = median(fixTimesBiased);
medianUnbiasedWhenBiasedLongestFix = median(fixTimeUnbiased);
meanBiasedWhenBiasedLongestFix = mean(fixTimesBiased);
meanUnbiasedWhenBiasedLongestFix = mean(fixTimeUnbiased);

[fixTimesBiased, fixTimeUnbiased] = ...
    FixTimesWhenBiasedLeastFixated(parcodes, sessions);
medianBiasedWhenBiasedLeastFix = median(fixTimesBiased);
medianUnbiasedWhenBiasedLeastFix = median(fixTimeUnbiased);
meanBiasedWhenBiasedLeastFix = mean(fixTimesBiased);
meanUnbiasedWhenBiasedLeastFix = mean(fixTimeUnbiased);

close all
